<?php
/**
 * @copyright  Vertex. All rights reserved.  https://www.vertexinc.com/
 * @author     Mediotype                     https://www.mediotype.com/
 */

namespace Vertex\Tax\Model\TaxInvoice;

use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Creditmemo;
use Magento\Sales\Model\Order\Invoice;
use Vertex\Data\LineItemInterface;
use Vertex\Data\LineItemInterfaceFactory;
use Vertex\Tax\Model\Config;
use Vertex\Tax\Model\ModuleManager;
use Vertex\Tax\Model\Repository\TaxClassNameRepository;

/**
 * Provide most Formatted data for an Invoice Request
 */
class BaseFormatter
{
    const TRANSACTION_TYPE = 'SALE';

    /** @var Config */
    private $config;

    /** @var LineItemInterfaceFactory */
    private $lineItemFactory;

    /** @var ModuleManager */
    private $moduleManager;

    /** @var TaxClassNameRepository */
    private $taxClassNameRepository;

    /**
     * @param Config $config
     * @param ModuleManager $moduleManager
     * @param TaxClassNameRepository $taxClassNameRepository
     * @param LineItemInterfaceFactory $lineItemFactory
     */
    public function __construct(
        Config $config,
        ModuleManager $moduleManager,
        TaxClassNameRepository $taxClassNameRepository,
        LineItemInterfaceFactory $lineItemFactory
    ) {
        $this->config = $config;
        $this->moduleManager = $moduleManager;
        $this->taxClassNameRepository = $taxClassNameRepository;
        $this->lineItemFactory = $lineItemFactory;
    }

    /**
     * Add Refund Adjustments to all line items if this is a Creditmemo
     *
     * @param LineItemInterface[] $items
     * @param Creditmemo $creditmemoModel
     * @return LineItemInterface[]
     */
    public function addRefundAdjustments($items, $creditmemoModel)
    {
        if (!($creditmemoModel instanceof Creditmemo)) {
            return $items;
        }

        if ($creditmemoModel->getBaseAdjustmentPositive()) {
            /** @var LineItemInterface $item */
            $item = $this->lineItemFactory->create();

            $item->setProductCode(
                $this->reduceCode(
                    $this->config->getCreditmemoAdjustmentPositiveCode(
                        $creditmemoModel->getStoreId()
                    )
                )
            );
            $item->setProductClass(
                $this->reduceCode(
                    $this->taxClassNameRepository->getById(
                        $this->config->getCreditmemoAdjustmentPositiveClass($creditmemoModel->getStoreId())
                    )
                )
            );

            $item->setQuantity(1);
            $item->setUnitPrice(-1 * $creditmemoModel->getBaseAdjustmentPositive());
            $item->setExtendedPrice($item->getUnitPrice());

            $items[] = $item;
        }

        if ($creditmemoModel->getBaseAdjustmentNegative()) {
            /** @var LineItemInterface $item */
            $item = $this->lineItemFactory->create();

            $item->setProductCode(
                $this->reduceCode(
                    $this->config->getCreditmemoAdjustmentFeeCode($creditmemoModel->getStoreId())
                )
            );
            $item->setProductClass(
                $this->reduceCode(
                    $this->taxClassNameRepository->getById(
                        $this->config->getCreditmemoAdjustmentFeeClass($creditmemoModel->getStoreId())
                    )
                )
            );
            $item->setQuantity(1);
            $item->setUnitPrice($creditmemoModel->getBaseAdjustmentNegative());
            $item->setExtendedPrice($item->getUnitPrice());

            $items[] = $item;
        }

        return $items;
    }

    /**
     * Get properly formatted order-level giftwrapping line-item data for an Invoice Request
     *
     * @param Order $order
     * @param Order|Invoice|Creditmemo $originalEntity
     * @param string|null $event
     * @return LineItemInterface
     */
    public function getFormattedOrderGiftWrap($order, $originalEntity = null, $event = null)
    {
        if (!$this->moduleManager->isEnabled('Magento_GiftWrapping')) {
            return null;
        }

        if ($originalEntity !== null && !$this->isFirstOfPartial($originalEntity)) {
            return null;
        }

        /** @var LineItemInterface $item */
        $item = $this->lineItemFactory->create();

        $item->setProductCode(
            $this->reduceCode(
                $this->config->getGiftWrappingOrderCode($order->getStoreId())
            )
        );
        $item->setProductClass(
            $this->reduceCode(
                $this->taxClassNameRepository->getById(
                    $this->config->getGiftWrappingOrderClass($order->getStoreId())
                )
            )
        );
        $item->setQuantity(1);
        $item->setUnitPrice($order->getGwBasePrice() ?: 0);

        if ($event === 'cancel' || $event === 'refund') {
            $item->setUnitPrice($item->getUnitPrice() * -1);
        }

        $item->setExtendedPrice($item->getUnitPrice());

        return $item;
    }

    /**
     * Get properly formatted Order-level Printed Card line item data for an Invoice Request
     *
     * @param Order $order
     * @param Order|Invoice|Creditmemo $originalEntity
     * @param string|null $event
     * @return LineItemInterface
     */
    public function getFormattedOrderPrintCard($order, $originalEntity = null, $event = null)
    {
        if (!$this->moduleManager->isEnabled('Magento_GiftWrapping')) {
            return null;
        }

        if ($originalEntity !== null && !$this->isFirstOfPartial($originalEntity)) {
            return null;
        }

        /** @var LineItemInterface $item */
        $item = $this->lineItemFactory->create();

        $item->setProductCode(
            $this->reduceCode(
                $this->config->getPrintedGiftcardCode($order->getStoreId())
            )
        );
        $item->setProductClass(
            $this->reduceCode(
                $this->taxClassNameRepository->getById(
                    $this->config->getPrintedGiftcardClass($order->getStoreId())
                )
            )
        );
        $item->setQuantity(1);
        $item->setUnitPrice($order->getGwCardBasePrice() ?: 0);

        if ($event === 'cancel' || $event === 'refund') {
            $item->setUnitPrice($item->getUnitPrice() * -1);
        }

        $item->setExtendedPrice($item->getUnitPrice());

        return $item;
    }

    /**
     * Get properly formatted Shipping Data for a Tax Invoice request
     *
     * @param Order|Invoice|Creditmemo $originalEntity
     * @param string|null $event
     * @return LineItemInterface
     */
    public function getFormattedShippingData($originalEntity = null, $event = null)
    {
        $order = $originalEntity instanceof Order ? $originalEntity : $originalEntity->getOrder();

        if (!$order || !$order->getShippingMethod() || !$this->isFirstOfPartial($originalEntity)) {
            return null;
        }

        /** @var LineItemInterface $item */
        $item = $this->lineItemFactory->create();

        $item->setProductCode($this->reduceCode($order->getShippingMethod()));
        $item->setProductClass(
            $this->reduceCode(
                $this->taxClassNameRepository->getById(
                    $this->config->getShippingTaxClassId($originalEntity->getStoreId())
                )
            )
        );
        if ($originalEntity instanceof Creditmemo) {
            $item->setUnitPrice(
                $originalEntity->getBaseShippingAmount() - $originalEntity->getBaseShippingDiscountAmount()
            );
        } else {
            $item->setUnitPrice($order->getBaseShippingAmount() - $order->getBaseShippingDiscountAmount());
        }
        $item->setQuantity(1);
        $item->setExtendedPrice($item->getUnitPrice());

        if ($event === 'cancel' || $event === 'refund') {
            $item->setUnitPrice($item->getUnitPrice() * -1);
            $item->setExtendedPrice($item->getExtendedPrice() * -1);
        }

        return $item;
    }

    /**
     * Determine if this is the first of a series of partial invoices
     *
     * @param Order|Invoice|Creditmemo $originalEntity
     * @return bool
     */
    private function isFirstOfPartial($originalEntity)
    {
        if ($originalEntity instanceof Invoice) {
            if (!$originalEntity->getBaseShippingTaxAmount()) {
                return false;
            }
        }

        if ($originalEntity instanceof Order && $originalEntity->getBaseShippingInvoiced() &&
            $this->config->requestByInvoiceCreation($originalEntity->getStore())) {
            return false;
        }

        if ($originalEntity instanceof Creditmemo) {
            if (!$originalEntity->getBaseShippingAmount()) {
                return false;
            }
        }

        return true;
    }

    /**
     * Reduce tax code to the maximum allowed characters
     *
     * @param string $code
     * @return string
     */
    private function reduceCode($code)
    {
        return substr($code, 0, Config::MAX_CHAR_PRODUCT_CODE_ALLOWED);
    }
}
