<?php
/**
 * @copyright  Vertex. All rights reserved.  https://www.vertexinc.com/
 * @author     Mediotype                     https://www.mediotype.com/
 */

namespace Vertex\Tax\Model\Api\Data;

use Vertex\Data\SellerInterface;
use Vertex\Data\SellerInterfaceFactory;
use Vertex\Tax\Model\Config;

/**
 * Create a {@see SellerInterface} from store configuration
 */
class SellerBuilder
{
    /** @var AddressBuilder */
    private $addressBuilder;

    /** @var Config */
    private $config;

    /** @var SellerInterfaceFactory */
    private $sellerFactory;

    /** @var string */
    private $storeCode;

    /**
     * @param SellerInterfaceFactory $sellerFactory
     * @param Config $config
     * @param AddressBuilder $addressBuilder
     */
    public function __construct(SellerInterfaceFactory $sellerFactory, Config $config, AddressBuilder $addressBuilder)
    {
        $this->sellerFactory = $sellerFactory;
        $this->config = $config;
        $this->addressBuilder = $addressBuilder;
    }

    /**
     * Create a {@see SellerInterface} from store configuration
     *
     * @return SellerInterface
     */
    public function build()
    {
        /** @var SellerInterface $seller */
        $seller = $this->sellerFactory->create();

        $street = [
            $this->config->getCompanyStreet1($this->storeCode),
            $this->config->getCompanyStreet2($this->storeCode)
        ];

        $address = $this->addressBuilder
            ->setStreet($street)
            ->setCity($this->config->getCompanyCity($this->storeCode))
            ->setRegionId($this->config->getCompanyRegionId($this->storeCode))
            ->setPostalCode($this->config->getCompanyPostalCode($this->storeCode))
            ->setCountryCode($this->config->getCompanyCountry($this->storeCode))
            ->build();

        $seller->setPhysicalOrigin($address);

        if ($this->config->getCompanyCode()) {
            $seller->setCompanyCode($this->config->getCompanyCode());
        }

        return $seller;
    }

    /**
     * Set the Store Code
     *
     * @param string|null $storeCode
     * @return SellerBuilder
     */
    public function setStoreCode($storeCode)
    {
        $this->storeCode = $storeCode;
        return $this;
    }
}
