<?php
/**
 * Refer to LICENSE.txt distributed with the Temando Shipping module for notice of license
 */
namespace Temando\Shipping\ViewModel\Batch;

use Magento\Framework\DataObjectFactory;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Sales\Api\Data\OrderAddressInterfaceFactory;
use Magento\Sales\Api\Data\ShipmentItemInterface;
use Magento\Sales\Model\Order\Address;
use Magento\Sales\Model\Order\Shipment;
use Temando\Shipping\Model\BatchInterface;
use Temando\Shipping\Model\BatchProviderInterface;
use Temando\Shipping\ViewModel\DataProvider\BatchUrl;
use Temando\Shipping\ViewModel\DataProvider\OrderAddress as AddressRenderer;
use Temando\Shipping\ViewModel\DataProvider\OrderDate;
use Temando\Shipping\ViewModel\DataProvider\OrderUrl;

/**
 * View model for batch details page.
 *
 * @package Temando\Shipping\ViewModel
 * @author  Rhodri Davies <rhodri.davies@temando.com>
 * @license https://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 * @link    https://www.temando.com/
 */
class BatchDetails implements ArgumentInterface
{
    /**
     * @var BatchProviderInterface
     */
    private $batchProvider;

    /**
     * @var OrderDate
     */
    private $orderDate;

    /**
     * @var BatchUrl
     */
    private $batchUrl;

    /**
     * @var OrderUrl
     */
    private $orderUrl;

    /**
     * @var DataObjectFactory
     */
    private $dataObjectFactory;

    /**
     * @var AddressRenderer
     */
    private $addressRenderer;

    /**
     * @var OrderAddressInterfaceFactory
     */
    private $addressFactory;

    /**
     * BatchDetails constructor.
     * @param BatchProviderInterface $batchProvider
     * @param OrderDate $orderDate
     * @param BatchUrl $batchUrl
     * @param OrderUrl $orderUrl
     * @param DataObjectFactory $dataObjectFactory
     * @param AddressRenderer $addressRenderer
     * @param OrderAddressInterfaceFactory $orderAddressInterfaceFactory
     */
    public function __construct(
        BatchProviderInterface $batchProvider,
        OrderDate $orderDate,
        BatchUrl $batchUrl,
        OrderUrl $orderUrl,
        DataObjectFactory $dataObjectFactory,
        AddressRenderer $addressRenderer,
        OrderAddressInterfaceFactory $orderAddressInterfaceFactory
    ) {
        $this->batchProvider = $batchProvider;
        $this->orderDate = $orderDate;
        $this->batchUrl = $batchUrl;
        $this->orderUrl = $orderUrl;
        $this->dataObjectFactory = $dataObjectFactory;
        $this->addressRenderer = $addressRenderer;
        $this->addressFactory = $orderAddressInterfaceFactory;
    }

    /**
     * @return \Magento\Framework\DataObject|null
     */
    public function getBatch()
    {
        /** @var \Temando\Shipping\Model\Batch $batch */
        $batch = $this->batchProvider->getBatch();
        return $batch;
    }

    /**
     * @param string $date
     * @return \DateTime
     */
    public function getDate(string $date): \DateTime
    {
        return $this->orderDate->getDate($date);
    }

    /**
     * Obtain batch create url
     *
     * @return string
     */
    public function getNewActionUrl(): string
    {
        return $this->batchUrl->getNewActionUrl();
    }

    /**
     * Obtain batch listing url
     *
     * @return string
     */
    public function getListActionUrl(): string
    {
        return $this->batchUrl->getListActionUrl();
    }

    /**
     * Obtain url for troubleshooting failed batches
     *
     * @return string
     */
    public function getSolveUrl(): string
    {
        return $this->batchUrl->getSolveActionUrl([
            BatchInterface::BATCH_ID => $this->batchProvider->getBatch()->getBatchId(),
        ]);
    }

    /**
     * @param string $orderId
     * @return string
     */
    public function getOrderViewUrl(string $orderId): string
    {
        return $this->orderUrl->getViewActionUrl(['order_id' => $orderId]);
    }

    /**
     * @param $extShipmentId
     * @return  \Magento\Sales\Api\Data\OrderInterface | null
     */
    private function getSalesShipmentOrder($extShipmentId)
    {
        $orders = $this->batchProvider->getOrders();
        if (isset($orders[$extShipmentId])) {
            return $orders[$extShipmentId];
        }

        return null;
    }

    /**
     * Returns ShipmentInfo based on extShipmentId.
     *
     * fixme(nr): replace by some proper piece of software
     *
     * @param string $extShipmentId
     * @return \Magento\Framework\DataObject
     */
    public function getShipmentInfoForGrid($extShipmentId)
    {
        $info = [];
        $order = $this->getSalesShipmentOrder($extShipmentId);
        if ($order) {
            /** @var Shipment $shipment */
            $shipment = $order->getShipmentsCollection()->getFirstItem();
            /** @var Address $shippingAddress */
            $shippingAddress = $shipment->getShippingAddress();
            $formattedShippingAddress = $this->addressRenderer->getFormattedAddress($shipment->getShippingAddress());
            $itemsInfo = $this->getItemsOrderedInfo($shipment->getItems());
            $info = [
                'shipment_id' => $shipment->getId(),
                'ship_to_name' => $shippingAddress->getFirstname() . ' ' . $shippingAddress->getLastname(),
                'destination_address' => $formattedShippingAddress,
                'items_ordered' => $itemsInfo,
            ];
        }

        $shipmentInfo = $this->dataObjectFactory->create(['data' => $info]);

        return $shipmentInfo;
    }

    /**
     * @param ShipmentItemInterface[] $items
     * @return string
     */
    private function getItemsOrderedInfo($items)
    {
        $info = '';
        foreach ($items as $item) {
            $info .= $item->getName() . ' x ' . (int) $item->getQty() . '<br>';
        }

        return $info;
    }
}
