<?php
/**
 * Refer to LICENSE.txt distributed with the Temando Shipping module for notice of license
 */
namespace Temando\Shipping\Webservice\Processor\OrderOperation;

use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Quote\Model\Quote\Address\RateRequest;
use Temando\Shipping\Api\Data\Delivery\PickupLocationSearchResultInterfaceFactory;
use Temando\Shipping\Api\Data\Delivery\QuotePickupLocationInterface;
use Temando\Shipping\Api\Data\Order\ShippingExperienceInterface;
use Temando\Shipping\Model\Delivery\QuotePickupLocation;
use Temando\Shipping\Model\OrderInterface;
use Temando\Shipping\Model\ResourceModel\Delivery\PickupLocationSearchResult;
use Temando\Shipping\Webservice\Response\Type\OrderResponseTypeInterface;

/**
 * Temando Pickup Location Search Processor.
 *
 * Persist pickup location search result.
 *
 * @package Temando\Shipping\Webservice
 * @author  Christoph Aßmann <christoph.assmann@netresearch.de>
 * @license https://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 * @link    https://www.temando.com/
 */
class PickupLocationSearchProcessor implements RatesProcessorInterface
{
    /**
     * @var PickupLocationSearchResultInterfaceFactory
     */
    private $pickupLocationSearchResultFactory;

    /**
     * PickupLocationSearchProcessor constructor.
     * @param PickupLocationSearchResultInterfaceFactory $pickupLocationSearchResultFactory
     */
    public function __construct(PickupLocationSearchResultInterfaceFactory $pickupLocationSearchResultFactory)
    {
        $this->pickupLocationSearchResultFactory = $pickupLocationSearchResultFactory;
    }

    /**
     * Persist pickup locations from rates response.
     *
     * @param RateRequest $rateRequest
     * @param OrderInterface $requestType
     * @param OrderResponseTypeInterface $responseType
     * @return ShippingExperienceInterface[]
     * @throws CouldNotSaveException
     */
    public function postProcess(
        RateRequest $rateRequest,
        OrderInterface $requestType,
        OrderResponseTypeInterface $responseType
    ) {
        $shippingAddressId = $requestType->getPickupLocationSearchRequest()->getShippingAddressId();
        $pickupLocationId = $requestType->getPickupLocation()->getPickupLocationId();

        if ($shippingAddressId && !$pickupLocationId) {
            // persist pickup locations for a given search request
            $pickupLocations = (array) $responseType->getPickupLocations();

            /** @var QuotePickupLocation $pickupLocation */
            foreach ($pickupLocations as $pickupLocation) {
                $pickupLocation->setData(QuotePickupLocationInterface::RECIPIENT_ADDRESS_ID, $shippingAddressId);
            }

            /** @var PickupLocationSearchResult $collection */
            $collection = $this->pickupLocationSearchResultFactory->create();
            $collection->addFilter(QuotePickupLocationInterface::RECIPIENT_ADDRESS_ID, $shippingAddressId);
            $collection->walk('delete');

            $collection->setItems($pickupLocations);
            $collection->save();
        }

        return [];
    }
}
