<?php
/**
 * Refer to LICENSE.txt distributed with the Temando Shipping module for notice of license
 */
namespace Temando\Shipping\Rest\EntityMapper;

use Temando\Shipping\Api\Data\Delivery\QuotePickupLocationInterface;
use Temando\Shipping\Api\Data\Delivery\QuotePickupLocationInterfaceFactory;
use Temando\Shipping\Rest\Response\DataObject\Location;
use Temando\Shipping\Rest\Response\DataObject\OrderQualification;
use Temando\Shipping\Rest\Response\Fields\Location\OpeningHours;

/**
 * Map API data to application data object
 *
 * @package Temando\Shipping\Rest
 * @author  Christoph Aßmann <christoph.assmann@netresearch.de>
 * @license http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 * @link    http://www.temando.com/
 */
class PickupLocationsResponseMapper
{
    /**
     * @var ShippingExperiencesMapper
     */
    private $shippingExperiencesMapper;

    /**
     * @var QuotePickupLocationInterfaceFactory
     */
    private $pickupLocationFactory;

    /**
     * PickupLocationsResponseMapper constructor.
     * @param ShippingExperiencesMapper $shippingExperiencesMapper
     * @param QuotePickupLocationInterfaceFactory $pickupLocationFactory
     */
    public function __construct(
        ShippingExperiencesMapper $shippingExperiencesMapper,
        QuotePickupLocationInterfaceFactory $pickupLocationFactory
    ) {
        $this->shippingExperiencesMapper = $shippingExperiencesMapper;
        $this->pickupLocationFactory = $pickupLocationFactory;
    }

    /**
     * @param OpeningHours $apiHours
     * @return string[]
     *
     * Date Format: ["l" => ["from" => "H:i:sP", "to" => "H:i:sP"]]
     */
    private function mapOpeningHours(OpeningHours $apiHours)
    {
        // general opening hours
        $openingHours = [];
        // specific opening hours
        $openingHoursSpecification = [
            'openings' => [],
            'closures' => [],
        ];

        foreach ($apiHours->getDefault() as $item) {
            $dow = $item->getDayOfWeek();
            $openingHours[$dow] = [
                'from' => $item->getOpens(),
                'to' => $item->getCloses(),
            ];
        }

        foreach ($apiHours->getExceptions()->getOpenings() as $opening) {
            $openingHoursSpecification['openings'][] = [
                'description' => $opening->getDescription(),
                'from' => $opening->getFrom(),
                'to' => $opening->getTo(),
            ];
        }

        foreach ($apiHours->getExceptions()->getClosures() as $closure) {
            $openingHoursSpecification['closures'][] = [
                'description' => $closure->getDescription(),
                'from' => $closure->getFrom(),
                'to' => $closure->getTo(),
            ];
        }

        $openingHours = [
            'general' => $openingHours,
            'specific' => $openingHoursSpecification,
        ];

        return $openingHours;
    }

    /**
     * @param OrderQualification[]|Location[] $apiIncluded
     * @return QuotePickupLocationInterface[]
     */
    public function map(array $apiIncluded)
    {
        /** @var OrderQualification[] $apiQualifications */
        $apiQualifications = array_filter($apiIncluded, function ($includedResource) {
            return ($includedResource instanceof OrderQualification);
        });

        $pickupLocations = [];
        foreach ($apiQualifications as $apiQualification) {
            /** @var Location $location */
            $location = current($apiQualification->getLocations());
            $locationAddress = $location->getAttributes()->getAddress();
            $openingHours = $this->mapOpeningHours($location->getAttributes()->getOpeningHours());
            $shippingExperiences = $this->shippingExperiencesMapper->map(
                $apiQualification->getAttributes()->getExperiences()
            );

            $pickupLocation = $this->pickupLocationFactory->create(['data' => [
                QuotePickupLocationInterface::PICKUP_LOCATION_ID => $location->getId(),
                QuotePickupLocationInterface::NAME => $location->getAttributes()->getName(),
                QuotePickupLocationInterface::COUNTRY => $locationAddress->getCountryCode(),
                QuotePickupLocationInterface::REGION => $locationAddress->getAdministrativeArea(),
                QuotePickupLocationInterface::POSTCODE => $locationAddress->getPostalCode(),
                QuotePickupLocationInterface::CITY => $locationAddress->getLocality(),
                QuotePickupLocationInterface::STREET => $locationAddress->getLines(),
                QuotePickupLocationInterface::OPENING_HOURS => $openingHours,
                QuotePickupLocationInterface::SHIPPING_EXPERIENCES => $shippingExperiences,
            ]]);

            $pickupLocations[]= $pickupLocation;
        }

        return $pickupLocations;
    }
}
