<?php
/**
 * Refer to LICENSE.txt distributed with the Temando Shipping module for notice of license
 */
namespace Temando\Shipping\Model\Shipping\RateRequest;

use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\Data\AddressExtensionInterface;
use Magento\Quote\Api\Data\AddressExtensionInterfaceFactory;
use Magento\Quote\Model\Quote\Address\RateRequest;
use Magento\Quote\Model\ShippingAddressManagementInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Temando\Shipping\Api\Data\Delivery\CollectionPointSearchRequestInterfaceFactory;
use Temando\Shipping\Api\Data\Delivery\PickupLocationSearchRequestInterfaceFactory;
use Temando\Shipping\Api\Data\Delivery\QuoteCollectionPointInterfaceFactory;
use Temando\Shipping\Api\Data\Delivery\QuotePickupLocationInterfaceFactory;
use Temando\Shipping\Model\OrderInterfaceBuilder;
use Temando\Shipping\Model\ResourceModel\Repository\AddressRepositoryInterface;
use Temando\Shipping\Model\ResourceModel\Repository\CollectionPointSearchRepositoryInterface;
use Temando\Shipping\Model\ResourceModel\Repository\PickupLocationSearchRepositoryInterface;
use Temando\Shipping\Model\ResourceModel\Repository\QuoteCollectionPointRepositoryInterface;
use Temando\Shipping\Model\ResourceModel\Repository\QuotePickupLocationRepositoryInterface;

/**
 * Temando Rate Request Data Initializer.
 *
 * @package Temando\Shipping\Model
 * @author  Christoph Aßmann <christoph.assmann@netresearch.de>
 * @license https://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 * @link    https://www.temando.com/
 */
class RequestDataInitializer
{
    /**
     * @var Extractor
     */
    private $rateRequestExtractor;

    /**
     * @var CollectionPointSearchRepositoryInterface
     */
    private $collectionPointSearchRepository;

    /**
     * @var CollectionPointSearchRequestInterfaceFactory
     */
    private $collectionPointSearchRequestFactory;

    /**
     * @var QuoteCollectionPointRepositoryInterface
     */
    private $collectionPointRepository;

    /**
     * @var QuoteCollectionPointInterfaceFactory
     */
    private $collectionPointFactory;

    /**
     * @var PickupLocationSearchRepositoryInterface
     */
    private $pickupLocationSearchRepository;

    /**
     * @var PickupLocationSearchRequestInterfaceFactory
     */
    private $pickupLocationSearchRequestFactory;

    /**
     * @var QuotePickupLocationRepositoryInterface
     */
    private $pickupLocationRepository;

    /**
     * @var QuotePickupLocationInterfaceFactory
     */
    private $pickupLocationFactory;

    /**
     * @var ShippingAddressManagementInterface
     */
    private $addressManagement;

    /**
     * @var AddressRepositoryInterface
     */
    private $checkoutAddressRepository;

    /**
     * @var AddressExtensionInterfaceFactory
     */
    private $addressExtensionFactory;

    /**
     * @var OrderInterfaceBuilder
     */
    private $orderBuilder;

    /**
     * RequestDataInitializer constructor.
     * @param Extractor $rateRequestExtractor
     * @param CollectionPointSearchRepositoryInterface $collectionPointSearchRepository
     * @param CollectionPointSearchRequestInterfaceFactory $collectionPointSearchRequestFactory
     * @param QuoteCollectionPointRepositoryInterface $collectionPointRepository
     * @param QuoteCollectionPointInterfaceFactory $collectionPointFactory
     * @param PickupLocationSearchRepositoryInterface $pickupLocationSearchRepository
     * @param PickupLocationSearchRequestInterfaceFactory $pickupLocationSearchRequestFactory
     * @param QuotePickupLocationRepositoryInterface $pickupLocationRepository
     * @param QuotePickupLocationInterfaceFactory $pickupLocationFactory
     * @param ShippingAddressManagementInterface $addressManagement
     * @param AddressRepositoryInterface $checkoutAddressRepository
     * @param AddressExtensionInterfaceFactory $addressExtensionFactory
     * @param OrderInterfaceBuilder $orderBuilder
     */
    public function __construct(
        Extractor $rateRequestExtractor,
        CollectionPointSearchRepositoryInterface $collectionPointSearchRepository,
        CollectionPointSearchRequestInterfaceFactory $collectionPointSearchRequestFactory,
        QuoteCollectionPointRepositoryInterface $collectionPointRepository,
        QuoteCollectionPointInterfaceFactory $collectionPointFactory,
        PickupLocationSearchRepositoryInterface $pickupLocationSearchRepository,
        PickupLocationSearchRequestInterfaceFactory $pickupLocationSearchRequestFactory,
        QuotePickupLocationRepositoryInterface $pickupLocationRepository,
        QuotePickupLocationInterfaceFactory $pickupLocationFactory,
        ShippingAddressManagementInterface $addressManagement,
        AddressRepositoryInterface $checkoutAddressRepository,
        AddressExtensionInterfaceFactory $addressExtensionFactory,
        OrderInterfaceBuilder $orderBuilder
    ) {
        $this->rateRequestExtractor = $rateRequestExtractor;
        $this->collectionPointSearchRepository = $collectionPointSearchRepository;
        $this->collectionPointSearchRequestFactory = $collectionPointSearchRequestFactory;
        $this->collectionPointRepository = $collectionPointRepository;
        $this->collectionPointFactory = $collectionPointFactory;
        $this->pickupLocationSearchRepository = $pickupLocationSearchRepository;
        $this->pickupLocationSearchRequestFactory = $pickupLocationSearchRequestFactory;
        $this->pickupLocationRepository = $pickupLocationRepository;
        $this->pickupLocationFactory = $pickupLocationFactory;
        $this->addressManagement = $addressManagement;
        $this->checkoutAddressRepository = $checkoutAddressRepository;
        $this->addressExtensionFactory = $addressExtensionFactory;
        $this->orderBuilder = $orderBuilder;
    }

    /**
     * Prepare the request type for quoting the current cart,
     * i.e. collect relevant data from rate request and persistent storage.
     *
     * @param RateRequest $rateRequest
     * @return \Temando\Shipping\Model\OrderInterface
     */
    public function getQuotingData(RateRequest $rateRequest)
    {
        $shippingAddress = $this->rateRequestExtractor->getShippingAddress($rateRequest);

        try {
            $checkoutAddress = $this->checkoutAddressRepository->getByQuoteAddressId($shippingAddress->getId());
            $checkoutFields = $checkoutAddress->getServiceSelection();
        } catch (LocalizedException $e) {
            $checkoutFields = [];
        }

        $addressExtension = $shippingAddress->getExtensionAttributes();
        if ($addressExtension instanceof AddressExtensionInterface) {
            $addressExtension->setCheckoutFields($checkoutFields);
        } else {
            $addressExtension = $this->addressExtensionFactory->create(['data' => [
                'checkout_fields' => $checkoutFields,
            ]]);
        }
        $shippingAddress->setExtensionAttributes($addressExtension);

        try {
            $collectionPointSearchRequest = $this->collectionPointSearchRepository->get($shippingAddress->getId());
        } catch (LocalizedException $exception) {
            $collectionPointSearchRequest = $this->collectionPointSearchRequestFactory->create();
        }

        try {
            $collectionPoint = $this->collectionPointRepository->getSelected($shippingAddress->getId());
        } catch (LocalizedException $exception) {
            $collectionPoint = $this->collectionPointFactory->create();
        }

        try {
            $pickupLocationSearchRequest = $this->pickupLocationSearchRepository->get($shippingAddress->getId());
        } catch (LocalizedException $exception) {
            $pickupLocationSearchRequest = $this->pickupLocationSearchRequestFactory->create();
        }

        try {
            $pickupLocation = $this->pickupLocationRepository->getSelected($shippingAddress->getId());
        } catch (LocalizedException $exception) {
            $pickupLocation = $this->pickupLocationFactory->create();
        }

        // create remote order entity from rate request
        $this->orderBuilder->setRateRequest($rateRequest);
        $this->orderBuilder->setCollectionPointSearchRequest($collectionPointSearchRequest);
        $this->orderBuilder->setCollectionPoint($collectionPoint);
        $this->orderBuilder->setPickupLocationSearchRequest($pickupLocationSearchRequest);
        $this->orderBuilder->setPickupLocation($pickupLocation);

        /** @var \Temando\Shipping\Model\OrderInterface $order */
        $order = $this->orderBuilder->create();

        return $order;
    }

    /**
     * Prepare the request type for manifesting the current order,
     * i.e. collect relevant data from sales order and persistent storage.
     *
     * @param OrderInterface $order
     * @return \Temando\Shipping\Model\OrderInterface
     * @throws LocalizedException
     */
    public function getManifestationData(OrderInterface $order)
    {
        try {
            $shippingAddress = $this->addressManagement->get($order->getQuoteId());
        } catch (NoSuchEntityException $exception) {
            throw new LocalizedException(__('Shipping address not found.'));
        }

        $collectionPointSearchRequest = $this->collectionPointSearchRequestFactory->create();
        $pickupLocationSearchRequest = $this->pickupLocationSearchRequestFactory->create();

        try {
            $collectionPoint = $this->collectionPointRepository->getSelected($shippingAddress->getId());
        } catch (LocalizedException $exception) {
            $collectionPoint = $this->collectionPointFactory->create();
        }

        try {
            $pickupLocation = $this->pickupLocationRepository->getSelected($shippingAddress->getId());
        } catch (LocalizedException $exception) {
            $pickupLocation = $this->pickupLocationFactory->create();
        }

        // create remote order entity from rate request
        $this->orderBuilder->setOrder($order);

        $this->orderBuilder->setCollectionPointSearchRequest($collectionPointSearchRequest);
        $this->orderBuilder->setCollectionPoint($collectionPoint);

        $this->orderBuilder->setPickupLocationSearchRequest($pickupLocationSearchRequest);
        $this->orderBuilder->setPickupLocation($pickupLocation);

        /** @var \Temando\Shipping\Model\OrderInterface $order */
        $order = $this->orderBuilder->create();

        return $order;
    }
}
