<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-sales-rule
 * @version   1.0.8
 * @copyright Copyright (C) 2020 Mirasvit (https://mirasvit.com/)
 */



namespace Mirasvit\SalesRule\Rule;

use Mirasvit\SalesRule\Api\Data\RuleTypeInterface;

class EachXmGetYmType implements RuleTypeInterface
{
    private $context;

    public function __construct(
        Context $context
    ) {
        $this->context = $context;
    }

    public function getType()
    {
        return 'mst_each_x_m_get_y_m';
    }

    public function getLabel()
    {
        return 'For each $X spend, give $Y discount';
    }

    public function calculate($rule, $item, $qty)
    {
        $discountData = $this->context->discountDataFactory->create();

        $discountStep   = (int)$rule->getDiscountStep();
        $discountAmount = $rule->getDiscountAmount();
        $discountQty    = $rule->getDiscountQty();

        if ($discountStep <= 1) {
            return $discountData;
        }

        $items = $this->context->getMatchingItems($item->getQuote(), $rule);

        $totalAmount = 0;
        foreach ($items as $itm) {
            $totalAmount += $this->context->validator->getItemBasePrice($itm) * $itm->getQty();
        }

        $multiplier = floor($totalAmount / $discountStep);

        if ($discountQty) {
            $multiplier = max($multiplier, $discountQty);
        }
        $totalDiscountAmount = $multiplier * $discountAmount;

        if ($totalDiscountAmount <= 0.01) {
            return $discountData;
        }

        $relativeDiscount = $totalDiscountAmount / $totalAmount;

        $itemPrice = $this->context->validator->getItemPrice($item);
        $discountData->setAmount($qty * $itemPrice * $relativeDiscount);

        $baseItemPrice = $this->context->validator->getItemBasePrice($item);
        $discountData->setBaseAmount($qty * $baseItemPrice * $relativeDiscount);

        $itemOriginalPrice = $this->context->validator->getItemOriginalPrice($item);
        $discountData->setOriginalAmount($itemOriginalPrice * $relativeDiscount);

        $baseItemOriginalPrice = $this->context->validator->getItemBaseOriginalPrice($item);
        $discountData->setBaseOriginalAmount($baseItemOriginalPrice * $relativeDiscount);

        return $discountData;
    }
}