<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-sales-rule
 * @version   1.2.7
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\SalesRule\Rule;

use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\SalesRule\Model\Rule;
use Magento\SalesRule\Model\Rule\Action\Discount\Data as DiscountData;
use Mirasvit\SalesRule\Service\Evaluator\Cart;
use Mirasvit\SalesRule\Api\Data\RuleTypeInterface;

class MostCheapestType extends MostExpensiveType implements RuleTypeInterface
{
    public function getType(): string
    {
        return 'mst_most_cheapest';
    }

    public function getLabel(): string
    {
        return 'Discount for Most Cheapest product';
    }

    public function calculate(Rule $rule, AbstractItem $item, float $qty): DiscountData
    {
        $cheapestItem = $this->getCheapestItem($item->getQuote(), $rule);

        if ($cheapestItem !== $item) {
            return $this->context->discountDataFactory->create();
        }

        return $this->getDiscountData($rule, $cheapestItem, $qty);
    }

    /**
     * @return \Magento\Quote\Model\Quote\Item|false
     */
    private function getCheapestItem(Quote $quote, Rule $rule)
    {
        $items = $this->context->getMatchingItems($quote, $rule);

        $item = false;
        $min  = false;
        foreach ($items as $itm) {
            $price = $this->context->validator->getItemPrice($itm);
            if ($price && ($min === false || $min > $price)) {
                $min  = $price;
                $item = $itm;
            }
        }

        $quoteRuleIds = $quote->getAppliedRuleIds() ? explode(',', $quote->getAppliedRuleIds()) : [];
        /** @var string $ids */
        $ids = $item->getAppliedRuleIds();
        $itemRuleIds  = $ids ? explode(',', $ids) : [];

        if (count($itemRuleIds) && count($quoteRuleIds) > count($itemRuleIds)) {
            return false;
        }

        return $item;
    }

    /**
     * @return Cart[]|array
     */
    public function evaluate(float $discountAmount, float $discountQty, float $discountStep): array
    {
        if ($discountAmount == 0) {
            return [];
        }

        $discountManyQty = !$discountQty || $discountQty > 3 ? 3 : $discountQty;

        $this->evaluator->addCart()
            ->addItem('', '$$', 1, $discountAmount, 1);

        $this->evaluator->addCart()
            ->addItem('', '$$', 3, $discountAmount, $discountManyQty);

        $this->evaluator->addCart()
            ->addItem('', '$$$$', 1)
            ->addItem('', '$$', 3, $discountAmount, $discountManyQty);

        $this->evaluator->addCart()
            ->addItem('', '$', 1, $discountAmount, 1)
            ->addItem('', '$', 1);

        $this->evaluator->addCart()
            ->addItem('', '$$$$', 1)
            ->addItem('', '$$', 1, $discountAmount, 1)
            ->addItem('', '$$$$$$', 1);

        return $this->evaluator->getCarts();
    }
}
