<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-sales-rule
 * @version   1.2.7
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\SalesRule\Rule;

use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\SalesRule\Model\Rule;
use Magento\SalesRule\Model\Rule\Action\Discount\Data as DiscountData;
use Mirasvit\SalesRule\Service\Evaluator\Cart;
use Mirasvit\SalesRule\Api\Data\RuleTypeInterface;

class ExceptMostExpensiveType extends AbstractType implements RuleTypeInterface
{
    public function getType(): string
    {
        return 'mst_except_expensive';
    }

    public function getLabel(): string
    {
        return 'Discount for All except Most Expensive product';
    }

    public function calculate(Rule $rule, AbstractItem $item, float $qty): DiscountData
    {
        $notExpensiveItems = $this->getNotExpensiveItems($item->getQuote(), $rule);

        foreach ($notExpensiveItems as $itm) {
            if ($item->getId() == $itm->getId()) {
                return $this->getDiscountData($rule, $item, $qty);
            }
        }

        return $this->context->discountDataFactory->create();
    }

    protected function getDiscountData(Rule $rule, AbstractItem $item, float $qty): DiscountData
    {
        $discountData = $this->context->discountDataFactory->create();

        if (!$item) {
            return $discountData;
        }

        $discountAmount = min($rule->getDiscountAmount(), 100) / 100;
        $discountQty    = $rule->getDiscountQty();

        if ($discountAmount < 0.01) {
            return $discountData;
        }

        if ($discountQty && $qty > $discountQty) {
            $qty = $discountQty;
        }

        $itemPrice = $this->context->validator->getItemPrice($item);
        $discountData->setAmount($qty * $itemPrice * $discountAmount);

        $baseItemPrice = $this->context->validator->getItemBasePrice($item);
        $discountData->setBaseAmount($qty * $baseItemPrice * $discountAmount);

        $itemOriginalPrice = $this->context->validator->getItemOriginalPrice($item);
        $discountData->setOriginalAmount($qty * $itemOriginalPrice * $discountAmount);

        $baseItemOriginalPrice = $this->context->validator->getItemBaseOriginalPrice($item);
        $discountData->setBaseOriginalAmount($qty * $baseItemOriginalPrice * $discountAmount);

        return $discountData;
    }

    /**
     * @return AbstractItem[]
     */
    private function getNotExpensiveItems(Quote $quote, Rule $rule): array
    {
        $items = $this->context->getMatchingItems($quote, $rule);

        $mostExpensive = false;
        $max           = false;

        foreach ($items as $itm) {
            $price = $this->context->validator->getItemPrice($itm);

            if ($max === false || $price > $max) {
                $max           = $price;
                $mostExpensive = $itm;
            }
        }

        $items = array_filter($items, function ($item) use ($mostExpensive) {
            return $item->getId() !== $mostExpensive->getId();
        });

        return $items;
    }

    /**
     * @return Cart[]|array
     */
    public function evaluate(float $discountAmount, float $discountQty, float $discountStep): array
    {
        if ($discountAmount == 0) {
            return [];
        }

        $cartComment = 'If cart matches rule conditions';
        $itemComment = 'If item matches action conditions';

        $discountManyQty = !$discountQty || $discountQty > 3 ? 3 : $discountQty;

        $this->evaluator->addCart()
            ->addItem('', '$$', 1);

        $this->evaluator->addCart()
            ->addItem('', '$$', 3);

        $this->evaluator->addCart()
            ->addItem('', '$$$$', 1)
            ->addItem('', '$$', 3, $discountAmount, $discountManyQty);

        $this->evaluator->addCart()
            ->addItem('', '$', 1)
            ->addItem('', '$', 3, $discountAmount, $discountManyQty);

        $this->evaluator->addCart()
            ->addItem('', '$$$$', 3, $discountAmount, $discountManyQty)
            ->addItem('', '$$', 3, $discountAmount, $discountManyQty)
            ->addItem('', '$$$$$$', 1);

        return $this->evaluator->getCarts();
    }

    public function getDiscountType(): string
    {
        return '%';
    }
}
