<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-sales-rule
 * @version   1.2.4
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\SalesRule\Rule;


use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\SalesRule\Model\Rule;
use Magento\SalesRule\Model\Rule\Action\Discount\Data as DiscountData;
use Mirasvit\SalesRule\Api\Data\RuleTypeInterface;
use Mirasvit\SalesRule\Service\Evaluator\Cart;

class CheapestWithStepType extends AbstractType implements RuleTypeInterface
{

    public function getType(): string
    {
        return 'mst_cheapest_with_step';
    }

    public function getLabel(): string
    {
        return 'Discount for Cheapest products by step';
    }

    public function calculate(Rule $rule, AbstractItem $item, float $qty): DiscountData
    {
        $itemsDiscountConfig = $this->getItemsDiscountConfig($item->getQuote(), $rule);

        if (!$itemsDiscountConfig) {
            return $this->context->discountDataFactory->create();
        }

        foreach ($itemsDiscountConfig as $itemConfig) {
            if ($itemConfig['item']->getId() == $item->getId()) {
                return $this->getDiscountData($rule, $item, $itemConfig['discountQty']);
            }
        }

        return $this->context->discountDataFactory->create();
    }

    protected function getDiscountData(Rule $rule, AbstractItem $item, float $qty): DiscountData
    {
        $discountData = $this->context->discountDataFactory->create();

        if (!$item) {
            return $discountData;
        }

        $discountAmount = min($rule->getDiscountAmount(), 100) / 100;

        if ($discountAmount < 0.01) {
            return $discountData;
        }

        $itemPrice = $this->context->validator->getItemPrice($item);
        $discountData->setAmount($qty * $itemPrice * $discountAmount);

        $baseItemPrice = $this->context->validator->getItemBasePrice($item);
        $discountData->setBaseAmount($qty * $baseItemPrice * $discountAmount);

        $itemOriginalPrice = $this->context->validator->getItemOriginalPrice($item);
        $discountData->setOriginalAmount($qty * $itemOriginalPrice * $discountAmount);

        $baseItemOriginalPrice = $this->context->validator->getItemBaseOriginalPrice($item);
        $discountData->setBaseOriginalAmount($qty * $baseItemOriginalPrice * $discountAmount);

        return $discountData;
    }

    private function getItemsDiscountConfig(Quote $quote, Rule $rule): ?array
    {
        $items         = $this->context->getMatchingItems($quote, $rule);
        $totalItemsQty = 0;

        $sorted = [];

        foreach ($items as $item) {
            $price = $this->context->validator->getItemPrice($item);

            $sorted[] = [
                'item'  => $item,
                'price' => $price
            ];

            $totalItemsQty += $item->getQty();
        }

        if (!$rule->getDiscountStep()) {
            return null;
        }

        $maxDiscountQty = floor($totalItemsQty/$rule->getDiscountStep());

        if ($maxDiscountQty === 0) {
            return null;
        }

        $allowedDiscountQty = $rule->getDiscountQty() && $maxDiscountQty > $rule->getDiscountQty()
            ? $rule->getDiscountQty()
            : $maxDiscountQty;

        usort($sorted, function ($a, $b) {
            return $a['price'] < $b['price'] ? -1 : 1;
        });

        foreach ($sorted as $idx => $itemConfig) {
            $itemQty = $itemConfig['item']->getQty();

            $itemDiscountQty = $allowedDiscountQty > $itemQty
                ? $itemQty
                : $allowedDiscountQty;

            $itemConfig['discountQty'] = (float)$itemDiscountQty;
            unset($itemConfig['price']); // price no longer needed

            $sorted[$idx] = $itemConfig;

            $allowedDiscountQty -= $itemDiscountQty;
        }

        return $sorted;
    }

    /**
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     *
     * @return Cart[]|array
     */
    public function evaluate(float $discountAmount, float $discountQty, float $discountStep): array
    {
        if ($discountStep == 0 || $discountAmount == 0) {
            return [];
        }

        $itemQty = $discountStep - 1;
        $itemQty = $itemQty <= 0 ? 1 : $itemQty;

        $this->evaluator->addCart()
            ->addItem('', '', $itemQty);

        $itemQty = $discountStep * 3 + 1;

        $itemDiscountQty = floor($itemQty/$discountStep);
        $itemDiscountQty = $discountQty && $itemDiscountQty > $discountQty
            ? $discountQty
            : $itemDiscountQty;

        $this->evaluator->addCart()
            ->addItem('', '', $itemQty, $discountAmount, $itemDiscountQty);

        $items = [
            ['qty' => 3],
            ['qty' => $discountStep * 3 + 1],
            ['qty' => 2],
        ];

        $totalQty       = $items[0]['qty'] + $items[1]['qty'] + $items[2]['qty'];
        $maxDiscountQty = floor($totalQty/$discountStep);

        $allowedDiscountQty = $discountQty && $maxDiscountQty > $discountQty
            ? $discountQty
            : $maxDiscountQty;

        foreach ($items as $idx => $item) {
            $itmDiscountQty = $allowedDiscountQty > $item['qty']
                ? $item['qty']
                : $allowedDiscountQty;

            $item['discountQty']    = $itmDiscountQty;
            $item['discountAmount'] = $itmDiscountQty ? $discountAmount : 0;

            $items[$idx] = $item;

            $allowedDiscountQty -= $itmDiscountQty;
        }

        $this->evaluator->addCart()
            ->addItem('', '$$', $items[0]['qty'], $items[0]['discountAmount'], $items[0]['discountQty'])
            ->addItem('', '$$$$$$', $items[2]['qty'], $items[2]['discountAmount'], $items[2]['discountQty'])
            ->addItem('', '$$$$', $items[1]['qty'], $items[1]['discountAmount'], $items[1]['discountQty']);

        return $this->evaluator->getCarts();
    }

    public function getDiscountType(): string
    {
        return '%';
    }
}
