<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-sales-rule
 * @version   1.1.1
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\SalesRule\Rule;

use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\SalesRule\Model\Rule;
use Magento\SalesRule\Model\Rule\Action\Discount\Data as DiscountData;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Mirasvit\SalesRule\Api\Data\RuleTypeInterface;
use Mirasvit\SalesRule\Service\Evaluator\Cart;
use Mirasvit\SalesRule\Service\EvaluatorService;

class EachXmGetYmType extends AbstractType implements RuleTypeInterface
{
    private $priceCurrency;

    public function __construct(
        PriceCurrencyInterface $priceCurrency,
        EvaluatorService $evaluator,
        Context $context
    ) {
        $this->priceCurrency = $priceCurrency;

        parent::__construct($evaluator, $context);
    }

    public function getType(): string
    {
        return 'mst_each_x_m_get_y_m';
    }

    public function getLabel(): string
    {
        return 'For each $X spend, give $Y discount';
    }

    public function calculate(Rule $rule, AbstractItem $item, float $qty): DiscountData
    {
        $discountData = $this->context->discountDataFactory->create();

        $discountStep   = (int)$rule->getDiscountStep();
        $discountAmount = (float)$rule->getDiscountAmount();
        $discountQty    = $rule->getDiscountQty();

        if ($discountStep <= 1) {
            return $discountData;
        }

        $items       = $this->context->getMatchingItems($item->getQuote(), $rule);
        $totalAmount = 0;

        foreach ($items as $itm) {
            $totalAmount += $this->context->validator->getItemBasePrice($itm) * $itm->getQty();
        }

        $totalAmount = min($totalAmount, $item->getQuote()->getBaseSubtotal());

        $multiplier = floor($totalAmount / $discountStep);

        if ($discountQty) {
            $multiplier = max($multiplier, $discountQty);
        }
        $totalDiscountAmount = $multiplier * $discountAmount;

        if ($totalDiscountAmount <= 0.01) {
            return $discountData;
        }

        $itemPrice             = $this->context->validator->getItemPrice($item);
        $baseItemPrice         = $this->context->validator->getItemBasePrice($item);
        $itemOriginalPrice     = $this->context->validator->getItemOriginalPrice($item);
        $baseItemOriginalPrice = $this->context->validator->getItemBaseOriginalPrice($item);

        $ruleTotals = 0;
        $quote      = $item->getQuote();
        $address    = $item->getAddress();
        $cartRules  = $address->getCartFixedRules();

        if (!isset($cartRules[$rule->getId()])) {
            $cartRules[$rule->getId()] = $totalDiscountAmount;
        }

        $availableDiscountAmount = (float)$cartRules[$rule->getId()];
        $discountType            = 'CartFixed' . $rule->getId();

        if ($availableDiscountAmount > 0) {
            $store       = $quote->getStore();
            $quoteAmount = $this->priceCurrency->convert($availableDiscountAmount, $store);

            $baseDiscountAmount = min($baseItemPrice * $qty, $availableDiscountAmount);
            $baseDiscountAmount = $this->priceCurrency->round($baseDiscountAmount);

            $availableDiscountAmount   -= $baseDiscountAmount;
            $cartRules[$rule->getId()] = $availableDiscountAmount;

            $discountData->setAmount($this->priceCurrency->round(min($itemPrice * $qty, $quoteAmount)));
            $discountData->setBaseAmount($baseDiscountAmount);
            $discountData->setOriginalAmount(min($itemOriginalPrice * $qty, $quoteAmount));
            $discountData->setBaseOriginalAmount($this->priceCurrency->round($baseItemOriginalPrice));
        }
        $address->setCartFixedRules($cartRules);

        return $discountData;
    }

    /**
     * @return Cart[]|array
     */
    public function evaluate(float $discountAmount, float $discountQty, float $discountStep): array
    {
        $price1 = $discountStep - floor($discountStep * 0.5);

        $this->evaluator->addCart()
            ->addItem('', '$' . $price1, 1)
            ->setTotal('$' . $price1)
            ->setTotalDiscount('$' . $this->calculateDiscountTotal(
                $price1,
                $discountAmount,
                $discountQty,
                $discountStep
            ));

        $this->evaluator->addCart()
            ->addItem('', '$' . $discountStep, 3)
            ->setTotal('$' . ($discountStep*3))
            ->setTotalDiscount('$' . $this->calculateDiscountTotal(
                $discountStep*3,
                $discountAmount,
                $discountQty,
                $discountStep
            ));

        $price1 = $discountStep - $discountAmount;
        $price1 = $price1 <= 0 ? 1 : $price1;
        $price2 = $discountStep + $discountAmount;
        $total1 = $price1 * 3 + $price2 * 2;

        $this->evaluator->addCart()
            ->addItem('', '$' . $price1, 3)
            ->addItem('', '$' . $price2, 2)
            ->setTotal('$' . $total1)
            ->setTotalDiscount('$' . $this->calculateDiscountTotal(
                    $total1,
                    $discountAmount,
                    $discountQty,
                    $discountStep
                ));

        return $this->evaluator->getCarts();
    }

    public function getDiscountType(): string
    {
        return '$';
    }

    private function calculateDiscountTotal(float $total, float $discountAmount, float $discountQty, float $discountStep): float
    {
        $calculatedDiscount = floor($total/$discountStep) * $discountAmount;

        if ($total < $discountStep) {
            return 0;
        } elseif (!$discountQty) {
            return $calculatedDiscount;
        } else {
            $discount = $calculatedDiscount > ($discountAmount * $discountQty)
                ? $discountAmount * $discountQty
                : $calculatedDiscount;



            return $discount > $total ? $total : $discount;
        }
    }
}
