<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-sales-rule
 * @version   1.1.1
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\SalesRule\Rule;

use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Item\AbstractItem;
use Magento\SalesRule\Model\Rule;
use Magento\SalesRule\Model\Rule\Action\Discount\Data as DiscountData;
use Mirasvit\SalesRule\Service\Evaluator\Cart;
use Mirasvit\SalesRule\Api\Data\RuleTypeInterface;

class BuyXGetYType extends AbstractType implements RuleTypeInterface
{
    public function getType(): string
    {
        return 'mst_buy_x_get_y';
    }

    public function getLabel(): string
    {
        return 'Buy product X Get % discount for product Y';
    }

    public function calculate(Rule $rule, AbstractItem $item, float $qty): DiscountData
    {
        $discountData = $this->context->discountDataFactory->create();

        $discountAmount = min($rule->getDiscountAmount(), 100) / 100;
        $discountQty = $rule->getDiscountQty();

        $conditions = $rule->getConditions()->asArray();

        if ($discountAmount < 0.01 || !isset($conditions['conditions'])) {
            return $discountData;
        }

        if ($discountQty && $qty > $discountQty) {
            $qty = $discountQty;
        }

        $itemPrice = $this->context->validator->getItemPrice($item);
        $discountData->setAmount($qty * $itemPrice * $discountAmount);

        $baseItemPrice = $this->context->validator->getItemBasePrice($item);
        $discountData->setBaseAmount($qty * $baseItemPrice * $discountAmount);

        $itemOriginalPrice = $this->context->validator->getItemOriginalPrice($item);
        $discountData->setOriginalAmount($qty * $itemOriginalPrice * $discountAmount);

        $baseItemOriginalPrice = $this->context->validator->getItemBaseOriginalPrice($item);
        $discountData->setBaseOriginalAmount($qty * $baseItemOriginalPrice * $discountAmount);

        return $discountData;
    }

    /**
     * @return Cart[]|array
     */
    public function evaluate(float $discountAmount, float $discountQty, float $discountStep): array
    {
        $discountManyQty = !$discountQty || $discountQty > 3 ? 3 : $discountQty;

        $this->evaluator->addCart()
            ->addItem(
                'Product X',
                '',
                3,
                $discountAmount,
                $discountManyQty,
                'If product matches Product X (rule) and Product Y (action) conditions'
            );

        $this->evaluator->addCart()
            ->addItem('Product X', '', 0, 0, 0)
            ->addItem('Product Y', '', 1, $discountAmount, 1);

        $this->evaluator->addCart()
            ->addItem('Product X', '', 0, 0, 0)
            ->addItem('Product Z', '', 0, 0, 0)
            ->addItem('Product Y', '', 3, $discountAmount, $discountManyQty);

        $this->evaluator->addCart()
            ->addItem('Product Y', '', 0, 0, 0)
            ->addItem('Product Z', '', 0, 0, 0)
            ->setComment('No products in the cart match Product X (rule) conditions');

        return $this->evaluator->getCarts();
    }

    public function getDiscountType(): string
    {
        return '%';
    }
}
