<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-rma
 * @version   2.2.21
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\Rma\Ui\QuickDataBar;

use Magento\Backend\Block\Template;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Select;
use Mirasvit\Core\Ui\QuickDataBar\MultiCellDataBlock;
use Mirasvit\Rma\Api\Data\RmaInterface;
use Mirasvit\Rma\Api\Repository\StatusRepositoryInterface;

class OwnerDataBlock extends MultiCellDataBlock
{
    private $resource;

    private $statusRepository;

    public function __construct(
        ResourceConnection $resource,
        StatusRepositoryInterface $statusRepository,
        Template\Context   $context
    ) {
        $this->resource         = $resource;
        $this->statusRepository = $statusRepository;

        parent::__construct($context);
    }

    public function getCode(): string
    {
        return 'rma_owners';
    }

    public function getProviderName(): string
    {
        return 'rma_listing.rma_listing.listing_top.listing_filters';
    }

    public function getFilterName(): string
    {
        return 'user_id';
    }

    public function getLabel(): string
    {
        return (string)__('Owners');
    }

    public function getChildDataBlocks(): array
    {
        return [];
    }

    public function getScalarValues(): array
    {
        $data = [];

        $select = $this->getSelect();

        $rows = $this->resource->getConnection()
            ->fetchAll($select);

        foreach ($rows as $row) {
            $data[] = [
                'label'  => $row['adminname'].':',
                'value'  => $row['value'],
                'filter' => $row['user_id'],
                'isLink' => true,
            ];
        }

        return $data;
    }

    public function getSelect(array $columns = []): Select
    {
        try {
            $status = $this->statusRepository->getByCode('closed');
        } catch (\Exception $e) {
            $status = false;
        }

        $columns = array_merge($columns, [
            'value'   => new \Zend_Db_Expr('COUNT(*)'),
            'user_id' => 'rma.user_id',
        ]);

        $select = $this->resource->getConnection()
            ->select()
            ->from(['rma' => $this->resource->getTableName(RmaInterface::TABLE_NAME)], $columns)
            ->joinLeft([
                'au' => $this->resource->getTableName('admin_user'),
            ], 'rma.user_id = au.user_id', new \Zend_Db_Expr('CONCAT(au.firstname, " ", au.lastname) as adminname'))
            ->group('rma.' . RmaInterface::KEY_USER_ID);

        if ($status) {
            $select->where('rma.status_id <>' . $status->getId());
        }

        return $select;
    }
}
