<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-rma
 * @version   2.2.7
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\Rma\Ui\QuickDataBar;

use Magento\Backend\Block\Template;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Select;
use Mirasvit\Core\Service\SerializeService;
use Mirasvit\Core\Ui\QuickDataBar\MultiCellDataBlock;
use Mirasvit\Rma\Api\Data\RmaInterface;
use Mirasvit\Rma\Api\Data\StatusInterface;

class StatusDataBlock extends MultiCellDataBlock
{
    private $resource;

    public function __construct(
        ResourceConnection $resource,
        Template\Context   $context
    ) {
        $this->resource = $resource;

        parent::__construct($context);
    }

    public function getCode(): string
    {
        return 'rma_status';
    }

    public function getProviderName(): string
    {
        return 'rma_listing.rma_listing.listing_top.listing_filters';
    }

    public function getFilterName(): string
    {
        return 'status_id';
    }

    public function getLabel(): string
    {
        return (string)__('In Progress');
    }

    public function getChildDataBlocks(): array
    {
        return [];
    }

    public function getScalarValues(): array
    {
        $data = [];

        $select = $this->getSelect();

        $rows = $this->resource->getConnection()
            ->fetchAll($select);

        foreach ($rows as $row) {
            $name   = SerializeService::decode($row['name']);
            $data[] = [
                'label'      => $name ? array_shift($name) . ':' : $row['name'] . ':',
                'value'      => $row['value'],
                'filter'     => $row['status_id'],
                'isLink'     => true,
                'labelClass' => 'mst-rma-badge status-branch-' . $row['color'],
            ];
        }

        return $data;
    }

    public function getSelect(array $columns = []): Select
    {
        $columns = array_merge($columns, [
            'value'     => new \Zend_Db_Expr('COUNT(*)'),
            'status_id' => 'rma.status_id',
        ]);

        return $this->resource->getConnection()
            ->select()
            ->from(['rma' => $this->resource->getTableName(RmaInterface::TABLE_NAME)], $columns)
            ->joinLeft([
                'rs' => $this->resource->getTableName(StatusInterface::TABLE_NAME),
            ], 'rma.status_id = rs.status_id', [
                'name'  => new \Zend_Db_Expr('rs.name'),
                'color' => 'rs.color',
            ])
            ->where('code <> "closed"')
            ->group('rma.' . RmaInterface::KEY_STATUS_ID);
    }
}
