<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-rma
 * @version   2.0.61-beta
 * @copyright Copyright (C) 2019 Mirasvit (https://mirasvit.com/)
 */



namespace Mirasvit\Rma\Block\Rma;

use Mirasvit\Rma\Api\Data\StatusInterface;
use Mirasvit\Rma\Api\Repository\StatusRepositoryInterface;

class View extends \Magento\Framework\View\Element\Template
{
    private   $currentStatusId;
    private   $registry;
    private   $statusRepository;
    private   $statusTree = [];
    private   $tmpStatusTree = [];

    protected $fieldManagement;
    protected $mailHelper;
    protected $messageManagement;
    protected $messageSearchManagement;
    protected $rmaConfig;
    protected $rmaHtmlHelper;
    protected $rmaManagement;
    protected $rmaOrderHtml;

    public function __construct(
        \Mirasvit\Rma\Api\Config\RmaConfigInterface $rmaConfig,
        \Mirasvit\Rma\Api\Service\Message\MessageManagementInterface $messageManagement,
        \Mirasvit\Rma\Api\Service\Message\MessageManagement\SearchInterface $messageSearchManagement,
        \Mirasvit\Rma\Helper\Mail $mailHelper,
        \Mirasvit\Rma\Helper\Order\Html $rmaOrderHtml,
        \Mirasvit\Rma\Helper\Rma\Html $rmaHtmlHelper,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\View\Element\Template\Context $context,
        \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface $rmaManagement,
        \Mirasvit\Rma\Api\Service\Field\FieldManagementInterface $fieldManagement,
        StatusRepositoryInterface $statusRepository,
        array $data = []
    ) {
        $this->rmaConfig               = $rmaConfig;
        $this->mailHelper              = $mailHelper;
        $this->messageManagement       = $messageManagement;
        $this->messageSearchManagement = $messageSearchManagement;
        $this->rmaOrderHtml            = $rmaOrderHtml;
        $this->rmaHtmlHelper           = $rmaHtmlHelper;
        $this->registry                = $registry;
        $this->rmaManagement           = $rmaManagement;
        $this->fieldManagement         = $fieldManagement;
        $this->statusRepository        = $statusRepository;

        parent::__construct($context, $data);
    }

    /**
     * {@inheritdoc}
     */
    protected function _prepareLayout()
    {
        parent::_prepareLayout();
        if ($rma = $this->getRma()) {
            $this->pageConfig->getTitle()->set(__('RMA #%1', $rma->getIncrementId()));
            $pageMainTitle = $this->getLayout()->getBlock('page.main.title');
            if ($pageMainTitle) {
                $pageMainTitle->setPageTitle(
                    __('RMA #%1 - %2', $rma->getIncrementId(),
                        $this->rmaManagement->getStatus($rma)->getName())
                );
            }
        }
    }

    /**
     * @return \Mirasvit\Rma\Api\Data\RmaInterface
     */
    public function getRma()
    {
        return $this->registry->registry('current_rma');
    }

    /**
     * @return string
     */
    public function getStatusMessage()
    {
        $rma = $this->getRma();
        $status = $this->rmaManagement->getStatus($rma);

        $message = $this->statusRepository->getHistoryMessageForStore($status, $rma->getStoreId());

        return $this->mailHelper->parseVariables($message, $rma);
    }

    /**
     * @return \Magento\Sales\Api\Data\OrderInterface[]|\Mirasvit\Rma\Model\OfflineOrder[]
     */
    public function getOrders()
    {
        return $this->rmaManagement->getOrders($this->getRma());
    }

    /**
     * @return void
     */
    private function resetTree()
    {
        $this->statusTree    = [];
        $this->tmpStatusTree = [];
    }

    /**
     * @param array $list
     * @param int   $parentId
     */
    private function createTree(&$list, $parentId, &$activeBranch, &$lockTree)
    {
        $this->tmpStatusTree[$parentId] = 1;
        if(isset($list[$parentId])) {
            $isLastChild = true;
            foreach ($list[$parentId] as $childId) {
                if ($childId > 0 && !isset($this->tmpStatusTree[$childId])) {
                    $isLastChild = false;
                    if ($childId == $this->getCurrentStatusId()) {
                        $activeBranch = true;
                        $findCurrentParent = false;
                        // remove nodes of previous branch
                        foreach ($this->tmpStatusTree as $k => $v) {
                            if ($findCurrentParent) {
                                unset($this->tmpStatusTree[$k]);
                            }
                            if ($k == $parentId) {
                                $findCurrentParent = true;
                            }
                        }
                    }
                    $this->createTree($list, $childId, $activeBranch, $lockTree);
                    // when found active branch, takes only first child and do not build other branches
                    if ($activeBranch) {
                        break;
                    }
                }
            }
            if ($isLastChild && $activeBranch) {
                $lockTree = true;
            }
        } elseif ($activeBranch) { // we found last child of current branch
            $lockTree = true;
        }
        if ($lockTree) {
            array_unshift($this->statusTree, $parentId);
        }
    }

    /**
     * @param array $ids
     * @return StatusInterface[]|\Mirasvit\Rma\Model\ResourceModel\Status\Collection
     */
    private function getChildrenStatuses($ids)
    {
        $statuses = $this->statusRepository->getCollection();
        $statuses->addActiveFilter();
        $statuses->getSelect()
            ->where('status_id in (?)', $ids)
            ->order(StatusInterface::KEY_IS_MAIN_BRANCH . ' desc')
            ->order(StatusInterface::KEY_SORT_ORDER . ' asc');

        return $statuses;
    }

    /**
     * @return array
     */
    public function getProgress()
    {
        $statuses = $this->statusRepository->getCollection();
        $statuses->addActiveFilter();
        $statuses->getSelect()->order(StatusInterface::KEY_SORT_ORDER . ' asc');

        $rmaStatusHistory = (array)explode(',', $this->getRma()->getStatusHistory());

        $parents = [];
        foreach ($statuses as $status) {
            // we need this query to sort children
            $childrenIds = [];
            foreach ($status->getChildrenIds() as $childrenId) {
                if (in_array($childrenId, $rmaStatusHistory) && !isset($parents[$childrenId])) {
                    $childrenIds[] = $childrenId;
                }
            }
            if (empty($childrenIds)) { // after upgrade RMA does not have history, so we use default
                $childrenIds = $status->getChildrenIds();
            }
            $children = $this->getChildrenStatuses($childrenIds);
            foreach ($children as $child) {
                $parents[$status->getId()][] = $child->getId();
            }
        }
        if (empty($parents)) { // after update to v2.0.61 statuses do not organized in tree yet, so we use old method
            $this->statusTree = $statuses->getAllIds();
        } else {
            $this->resetTree();
            $parentId = $this->rmaConfig->getDefaultStatus();
            $activeBranch = $parentId == $this->getCurrentStatusId();
            $lockTree = false;
            $this->createTree($parents, $parentId, $activeBranch, $lockTree);
        }

        $progress = [];
        foreach ($this->statusTree as $statusId) {
            $status = $statuses->getItemById($statusId);

            // v2.0.60 does not show "rejected" on frontend
            if (empty($parents) && $status->getCode() == 'rejected') {
                continue;
            }
            $progress[] = [
                'label'   => $status->getName(),
                'active'  => false,
                'visible' => $status->getIsVisible(),
            ];

            if ($status->getId() === $this->getCurrentStatusId()) {
                foreach (array_keys($progress) as $key) {
                    $progress[$key]['active'] = true;
                }
            }
        }

        return $progress;
    }

    /**
     * @return int
     */
    public function getCurrentStatusId()
    {
        if (!$this->currentStatusId) {
            $this->currentStatusId = $this->getRma()->getStatusId();
        }

        return $this->currentStatusId;
    }

    /**
     * @param \Magento\Sales\Api\Data\OrderInterface|\Mirasvit\Rma\Model\OfflineOrder $order
     *
     * @return string
     */
    public function getOrderLabel($order)
    {
        if ($order->getIsOffline()) {
            return $order->getReceiptNumber();
        } else {
            return '#' . $order->getIncrementId();
        }
    }
}
