<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-rma
 * @version   2.0.30
 * @copyright Copyright (C) 2018 Mirasvit (https://mirasvit.com/)
 */



namespace Mirasvit\Rma\Service\Rma\RmaManagement;

use Magento\Framework\Exception\LocalizedException;

class CreateReplacementOrder implements \Mirasvit\Rma\Api\Service\Rma\RmaManagement\CreateReplacementOrderInterface
{

    public function __construct(
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        \Magento\Customer\Model\CustomerFactory $customerFactory,
        \Magento\Customer\Api\CustomerRepositoryInterface $customerRepository,
        \Magento\Quote\Api\CartManagementInterface $cartManagement,
        \Magento\Quote\Api\CartRepositoryInterface $cartRepository,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Mirasvit\Rma\Api\Service\Item\ItemManagementInterface $itemManagement,
        \Mirasvit\Rma\Api\Service\Rma\RmaManagementInterface $rmaManagement,
        \Mirasvit\Rma\Api\Service\Rma\RmaManagement\SearchInterface $rmaSearchManagement
    ) {
        $this->productRepository = $productRepository;
        $this->customerFactory = $customerFactory;
        $this->customerRepository = $customerRepository;
        $this->cartManagement = $cartManagement;
        $this->cartRepository = $cartRepository;
        $this->storeManager = $storeManager;
        $this->itemManagement = $itemManagement;
        $this->rmaManagement = $rmaManagement;
        $this->rmaSearchManagement = $rmaSearchManagement;
    }

    /**
     * {@inheritdoc}
     */
    public function create(\Mirasvit\Rma\Api\Data\RmaInterface $rma)
    {
        if (!$customer = $rma->getCustomerId()) {
            throw new \Exception('Replacement Orders available only for registered customers!');
        }
        $originOrder = $this->rmaManagement->getOrder($rma);
        $orderStore = $this->storeManager->getStore($originOrder->getStoreId());
        $orderQuote = $this->cartRepository->get($originOrder->getQuoteId());
        $paymentMethod = $orderQuote->getPayment()->getMethod();

        $customer= $this->customerRepository->getById($originOrder->getCustomerId());

        $cartId = $this->cartManagement->createEmptyCart();
        $cart = $this->cartRepository->get($cartId);
        $cart->setStore($orderStore);
        $cart->setCurrency($orderQuote->getCurrency());
        $cart->assignCustomer($customer);

        $items = $this->getItems($rma);
        if (!count($items)) {
            throw new LocalizedException(
                __('At least one RMA item should have resolution "Exchange".')
            );
        }
        foreach ($items as $item) {
            if ($this->itemManagement->isExchange($item)) {
                $product = $this->productRepository->get($item->getProductSku());
                $product->setPrice(0);
                $cart->addProduct($product, $item->getQtyRequested());
            }
        }
        $cart->getBillingAddress()->addData($originOrder->getBillingAddress()->getData());
        if ($originOrder->getShippingAddress()) {
            $cart->getShippingAddress()->addData($originOrder->getShippingAddress()->getData());
            $cart->getShippingAddress()->setShippingMethod('freeshipping_freeshipping');
            // uncomment to set original shipping
//            $method = $orderQuote->getShippingAddress()->getShippingMethod();
//            $cart->getShippingAddress()->setShippingMethod($method);
        }

        $cart->setPaymentMethod($paymentMethod);
        $cart->setInventoryProcessed(false);
        $cart->getPayment()->importData(['method' => $paymentMethod]);

        // Collect total and save
        $cart->collectTotals();

        // Submit the quote and create the order
        $cart->save();
        $cart = $this->cartRepository->get($cart->getId());

        $orderId = $this->cartManagement->placeOrder($cart->getId());
        $replacementOrderIds = $rma->getReplacementOrderIds();
        $replacementOrderIds[] = $orderId;
        $rma->setReplacementOrderIds($replacementOrderIds);
        $rma->save();

        return $orderId;
    }

    /**
     * @param \Mirasvit\Rma\Api\Data\RmaInterface $rma
     * @return \Mirasvit\Rma\Api\Data\ItemInterface[]
     */
    private function getItems(\Mirasvit\Rma\Api\Data\RmaInterface $rma)
    {
        return $this->rmaSearchManagement->getRequestedItems($rma);
    }
}