<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-report
 * @version   1.3.5
 * @copyright Copyright (C) 2018 Mirasvit (https://mirasvit.com/)
 */



namespace Mirasvit\ReportApi\Config;

use Mirasvit\ReportApi\Api\Config\AggregatorInterface;
use Mirasvit\ReportApi\Api\Config\RelationInterface;
use Mirasvit\ReportApi\Api\Config\TableInterface;
use Mirasvit\ReportApi\Api\SchemaInterface;
use Mirasvit\ReportApi\Config\Loader\MapFactory;

class Schema implements SchemaInterface
{
    /**
     * @var MapFactory
     */
    private $mapFactory;

    /**
     * @var TableInterface[]
     */
    private $tablePool = [];

    /**
     * @var RelationInterface[]
     */
    private $relationPool = [];

    /**
     * @var string[string]
     */
    private $typePool = [];

    /**
     * @var string[string]
     */
    private $aggregatorPool;

    public function __construct(
        MapFactory $mapFactory,
        array $type = [],
        array $aggregator = []
    ) {
        $this->mapFactory = $mapFactory;
        $this->typePool = $type;
        $this->aggregatorPool = $aggregator;
    }

    private function initialize()
    {
        if (!$this->tablePool) {
            $this->mapFactory->create()
                ->load();
        }

        return $this;
    }

    public function getType($type)
    {
        //class type
        if (strpos($type, '\\') !== false) {
            return $type;
        }

        //predefined type
        if (isset($this->typePool[$type])) {
            return $this->typePool[$type];
        }

        throw new \Exception("Unsupported type $type");
    }

    public function getAggregator($aggregator)
    {
        if (isset($this->aggregatorPool[$aggregator])) {
            return $this->aggregatorPool[$aggregator];
        }

        throw new \Exception("Unsupported aggregator $aggregator");
    }

    /**
     * {@inheritdoc}
     */
    public function getTables()
    {
        $this->initialize();

        return $this->tablePool;
    }

    /**
     * {@inheritdoc}
     */
    public function getTable($tableName)
    {
        if (!key_exists($tableName, $this->getTables())) {
            throw new \Exception(__("Table '%1' is not defined.", $tableName));
        }

        $table = $this->getTables()[$tableName];

        return $table;
    }

    public function hasTable($tableName)
    {
        return key_exists($tableName, $this->getTables());
    }

    /**
     * {@inheritdoc}
     */
    public function getColumn($identifier)
    {
        if (count(explode('|', $identifier)) == 3) {
            list(, $tableName, $columnName) = explode('|', $identifier);
        } elseif (count(explode('|', $identifier)) == 2) {
            list($tableName, $columnName) = explode('|', $identifier);
        } else {
            throw new \Exception("Wrong column identifier: $identifier");
        }

        $column = $this->getTable($tableName)->getColumn("$columnName");

        return $column;
    }

    public function getSimpleColumns($table)
    {
        $result = [];

        foreach ($this->getTable($table)->getColumns() as $column) {
            if (!in_array($column->getAggregator()->getType(), [
                AggregatorInterface::TYPE_SUM,
                AggregatorInterface::TYPE_COUNT,
                AggregatorInterface::TYPE_AVERAGE,
                AggregatorInterface::TYPE_CONCAT,
            ])
            ) {
                $result[] = $column->getIdentifier();
            }
        }

        return $result;
    }

    public function getComplexColumns($table)
    {
        $result = [];

        foreach ($this->getTable($table)->getColumns() as $column) {
            if (in_array($column->getAggregator()->getType(), [
                AggregatorInterface::TYPE_SUM,
                AggregatorInterface::TYPE_COUNT,
                AggregatorInterface::TYPE_AVERAGE,
            ])) {
                $result[] = $column->getIdentifier();
            }
        }

        return $result;
    }

    /**
     * {@inheritdoc}
     */
    public function getRelations()
    {
        return $this->relationPool;
    }

    public function addTable(TableInterface $table)
    {
        $this->tablePool[$table->getName()] = $table;

        return $this;
    }

    public function addRelation(RelationInterface $relation)
    {
        $this->relationPool[] = $relation;

        return $this;
    }
}