<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.3.2
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\EmailReport\Service\MailModifier;

use Magento\Store\Api\Data\StoreInterface;
use Mirasvit\Email\Api\Data\QueueInterface;
use Mirasvit\EmailReport\Service\StorageService;
use Mirasvit\EmailDesigner\Service\TemplateEngine\Liquid\Variable\Context;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\UrlInterface;

class LinkModifier
{

    private $urlBuilder;

    private $context;

    private $storeManager;


    public function __construct(
        UrlInterface          $urlBuilder,
        StoreManagerInterface $storeManager,
        Context               $context
    ) {
        $this->urlBuilder   = $urlBuilder;
        $this->storeManager = $storeManager;
        $this->context      = $context;
    }

    public function modifyContent(QueueInterface $queue, string $content): string
    {
        $params = [
            StorageService::QUEUE_PARAM_NAME => $queue->getUniqHash(),
        ];

        if (!$store = $this->context->getStore()) {
            $store = $this->storeManager->getStore();
        }
        $storeUrl = $store->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_DIRECT_LINK);

        if (preg_match_all('/<a\s[^>]*href=([\"\']??)([^\" >]*?)\\1[^>]*>(.*)<\/a>/siU', $content, $matches)) {
            foreach ($matches[2] as $key => $url) {
                // modify only store urls
                if (!strstr($url, $storeUrl)) {
                    continue;
                }

                $newUrl = $this->createLink($store, $url, $params);

                if ($newUrl) {
                    $backendFrontName = $this->urlBuilder->getAreaFrontName();
                    if ($backendFrontName && strpos($newUrl, '/'.$backendFrontName.'/') !== false) {
                        $newUrl = str_replace('/'.$backendFrontName.'/', '/', $newUrl);
                    }

                    $from = $matches[0][$key];
                    $to = str_replace('href="' . $url . '"', 'href="' . $newUrl . '"', $from);

                    $content = str_replace($from, $to, $content);
                }
            }
        }

        return $content;
    }

    /**
     * @param StoreInterface $store
     * @param string         $url
     * @param array          $params
     *
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function createLink(StoreInterface $store, $url, array $params)
    {
        $params['_query'] = [
            'url' => base64_encode($url),
        ];

        return $store->getUrl('emailreport/track/click', $params);
    }
}
