<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.3.2
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Ui\Campaign\View;

use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\Search\ReportingInterface;
use Magento\Framework\Api\Search\SearchCriteriaBuilder;
use Magento\Framework\Api\Search\SearchResultInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Element\UiComponent\DataProvider\DataProvider as BaseDataProvider;
use Mirasvit\Email\Api\Data\CampaignInterface;
use Mirasvit\Email\Api\Data\ChainInterface;
use Mirasvit\Email\Api\Data\QueueInterface;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Repository\QueueRepository;
use Mirasvit\Email\Repository\Trigger\ChainRepository;
use Mirasvit\Email\Repository\TriggerRepository;
use Mirasvit\EmailDesigner\Api\Data\TemplateInterface;
use Mirasvit\EmailReport\Ui\ReportDataProvider;
use Mirasvit\Event\Api\Repository\EventRepositoryInterface;
use Mirasvit\Event\Model\Rule;
use Mirasvit\Event\Model\RuleFactory;

class DataProvider extends BaseDataProvider
{
    protected $reportDataProvider;

    protected $urlBuilder;

    protected $triggerRepository;

    protected $chainRepository;

    protected $eventRepository;

    protected $queueRepository;

    protected $ruleFactory;

    public function __construct(
        TriggerRepository        $triggerRepository,
        UrlInterface             $urlBuilder,
        ChainRepository          $chainRepository,
        EventRepositoryInterface $eventRepository,
        QueueRepository          $queueRepository,
        RuleFactory              $ruleFactory,
        string                   $name,
        string                   $primaryFieldName,
        string                   $requestFieldName,
        ReportingInterface       $reporting,
        SearchCriteriaBuilder    $searchCriteriaBuilder,
        RequestInterface         $request,
        FilterBuilder            $filterBuilder,
        ReportDataProvider       $reportDataProvider,
        array                    $meta = [],
        array                    $data = []
    ) {
        $this->reportDataProvider = $reportDataProvider;
        $this->chainRepository    = $chainRepository;
        $this->eventRepository    = $eventRepository;
        $this->queueRepository    = $queueRepository;
        $this->urlBuilder         = $urlBuilder;
        $this->ruleFactory        = $ruleFactory;
        $this->triggerRepository  = $triggerRepository;

        parent::__construct(
            $name,
            $primaryFieldName,
            $requestFieldName,
            $reporting,
            $searchCriteriaBuilder,
            $request,
            $filterBuilder,
            $meta,
            $data
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function searchResultToOutput(SearchResultInterface $searchResult)
    {
        $result = [
            'items' => [],
        ];

        foreach ($searchResult->getItems() as $item) {
            $data = [];
            foreach ($item->getCustomAttributes() as $attribute) {
                $data[$attribute->getAttributeCode()] = $attribute->getValue();
            }

            $data = $this->addCampainLinks($data);
            $data = $this->addTriggers($data);

            if ($this->reportDataProvider) {
                $data = $this->reportDataProvider->addReportData($data);
            } else {
                $data['report'] = null;
            }

            $result['items'][] = $data;
        }

        return $result;
    }

    protected function addCampainLinks(array $data): array
    {
        $data['view_url']      = $this->urlBuilder->getUrl(
            'email/campaign/view',
            [CampaignInterface::ID => $data[CampaignInterface::ID]]
        );
        $data['delete_url']    = $this->urlBuilder->getUrl(
            'email/campaign/delete',
            [CampaignInterface::ID => $data[CampaignInterface::ID]]
        );
        $data['duplicate_url'] = $this->urlBuilder->getUrl(
            'email/campaign/duplicate',
            [CampaignInterface::ID => $data[CampaignInterface::ID]]
        );

        return $data;
    }

    protected function addTriggers(array $data): array
    {
        $data['triggers'] = [];

        $collection = $this->triggerRepository->getCollection();
        $collection->addFieldToFilter(TriggerInterface::CAMPAIGN_ID, $data[CampaignInterface::ID]);

        foreach ($collection as $trigger) {
            $triggerData = [
                'id_field_name'         => TriggerInterface::ID,
                TriggerInterface::ID    => $trigger->getId(),
                TriggerInterface::TITLE => $trigger->getTitle(),
                'count'                 => $trigger->getChainCollection()->getSize(),
            ];

            $triggerData = $this->addTriggerExtraData($trigger, $triggerData);

            if ($this->reportDataProvider) {
                $triggerData = $this->reportDataProvider->addReportData($triggerData);
            } else {
                $triggerData['report'] = null;
            }

            $data['triggers'][] = $triggerData;
        }

        return $data;
    }

    private function addTriggerExtraData(TriggerInterface $trigger, array $data): array
    {
        $extraData = [
            TriggerInterface::IS_ACTIVE   => $trigger->getIsActive(),
            TriggerInterface::ACTIVE_FROM => $trigger->getActiveFrom(),
            TriggerInterface::ACTIVE_TO   => $trigger->getActiveTo(),
            TriggerInterface::DESCRIPTION => $trigger->getDescription(),
            TriggerInterface::EVENT       => $this->getEvent($trigger),
            TriggerInterface::RULE        => $this->getRule($trigger),
            'duplicate_url' => $this->urlBuilder->getUrl(
                'email/trigger/move',
                [
                    '_current'           => 1,
                    TriggerInterface::ID => $trigger->getId(),
                    '_query'             => [
                        'campaigns' => [$trigger->getCampaignId()],
                    ],
                ]
            ),
            'view_url'      => $this->urlBuilder->getUrl(
                'email/campaign/view',
                [
                    CampaignInterface::ID => $trigger->getCampaignId(),
                    '_fragment'           => TriggerInterface::ID . '_' . $trigger->getId(),
                ]
            ),
            'delete_url'    => $this->urlBuilder->getUrl(
                'email/trigger/delete',
                [TriggerInterface::ID => $trigger->getId()]
            ),
            'toggle_url'    => $this->urlBuilder->getUrl(
                'email/trigger/toggle',
                [TriggerInterface::ID => $trigger->getId()]
            ),
            'report' => [
                'pendingCount' => $this->countPendingEmails($trigger->getId()),
            ],
        ];

        $data = array_merge_recursive($data, $extraData);

        $data = $this->addChains($data);

        return $data;
    }

    private function addChains(array $data): array
    {
        $data['chains'] = [];

        $collection = $this->chainRepository->getCollection();
        $collection->addFieldToFilter(ChainInterface::TRIGGER_ID, $data[TriggerInterface::ID]);

        /** @var \Mirasvit\Email\Model\Trigger\Chain $chain */
        foreach ($collection as $chain) {
            $duplicateUrl = $this->urlBuilder->getUrl('email/chain/duplicate', [
                '_current'         => 1,
                ChainInterface::ID => $chain->getId(),
            ]);

            $deleteUrl = $this->urlBuilder->getUrl('email/chain/delete', [
                '_current' => 1,
                ChainInterface::ID => $chain->getId()
            ]);

            $chainData = [
                'id_field_name'          => ChainInterface::ID,
                ChainInterface::ID       => $chain->getId(),
                TemplateInterface::TITLE => $chain->getTemplate()
                    ? $chain->getTemplate()->getTitle()
                    : (string)__('No Template Selected'),
                'info'                   => (string)$chain->toString(),
                'duplicate_url'          => $duplicateUrl,
                'delete_url'             => $deleteUrl
            ];

            if ($this->reportDataProvider) {
                $chainData = $this->reportDataProvider->addReportData($chainData);
            } else {
                $chainData['report'] = null;
            }

            $data['chains'][] = $chainData;
        }

        return $data;
    }

    private function getRule(TriggerInterface $trigger): string
    {
        /** @var Rule $rule */
        $rule = $this->ruleFactory->create();
        $rule->loadPost($trigger->getRule());

        return $rule->toString();
    }

    private function getEvent(TriggerInterface $trigger): string
    {
        if (!$trigger->getEvent()) {
            return '';
        }

        // event was wrong or removed
        if (!$this->eventRepository->getInstance($trigger->getEvent())) {
            return '';
        }

        return (string)$this->eventRepository->getInstance($trigger->getEvent())
                           ->getEvents()[$trigger->getEvent()];
    }

    private function countPendingEmails(int $triggerId): int
    {
        $queues = $this->queueRepository->getCollection();
        $queues->addFieldToFilter(TriggerInterface::ID, $triggerId)
            ->addFieldToFilter(QueueInterface::STATUS, QueueInterface::STATUS_PENDING);

        return $queues->count();
    }
}
