<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.2
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\EmailDesigner\Model;

use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Registry;
use Mirasvit\EmailDesigner\Api\Data\TemplateInterface;
use Mirasvit\EmailDesigner\Api\Data\ThemeInterface;
use Mirasvit\Core\Service\YamlService as YamlParser;
use Mirasvit\Email\Helper\Serializer;

class Template extends AbstractModel implements TemplateInterface
{
    const TYPE = 'emaildesigner';

    protected $themeFactory;

    protected $config;

    protected $areas;

    protected $theme;

    protected $serializer;

    public function __construct(
        ThemeFactory $themeFactory,
        ConfigProvider $config,
        Context $context,
        Registry $registry,
        Serializer $serializer
    ) {
        $this->themeFactory = $themeFactory;
        $this->config       = $config;
        $this->serializer   = $serializer;

        parent::__construct($context, $registry);
    }

    /**
     * {@inheritdoc}
     */
    protected function _construct()
    {
        $this->_init(\Mirasvit\EmailDesigner\Model\ResourceModel\Template::class);
    }

    public function getId()
    {
        return $this->getData(self::ID) ? (int)$this->getData(self::ID) : null;
    }

    public function getTitle(): string
    {
        return (string)$this->getData(self::TITLE);
    }

    public function setTitle(string $title): TemplateInterface
    {
        $this->setData(self::TITLE, $title);

        return $this;
    }

    public function getDescription(): string
    {
        return (string)$this->getData(self::DESCRIPTION);
    }

    public function setDescription(string $description): TemplateInterface
    {
        $this->setData(self::DESCRIPTION, $description);

        return $this;
    }

    public function getTemplateSubject(): string
    {
        return (string)$this->getData(self::TEMPLATE_SUBJECT);
    }

    public function getTemplateText(): string
    {
        return (string)$this->getTheme()->getTemplateText();
    }

    public function setTemplateSubject(string $subject): TemplateInterface
    {
        $this->setData(self::TEMPLATE_SUBJECT, $subject);

        return $this;
    }

    public function getAreas(): array
    {
        if ($this->areas == null) {
            if ($this->getTheme()) {
                $this->areas = $this->getTheme()->getAreas();
            } else {
                $this->areas = ['content' => 'Content'];
            }
        }

        return $this->areas;
    }

    public function getTemplateAreas(): array
    {
        if (!$this->hasData(self::TEMPLATE_AREAS)) {
            $templateAreas = $this->serializer->unserialize($this->getTemplateAreasSerialized());
            if (is_array($templateAreas)) {
                $this->setData(self::TEMPLATE_AREAS, $templateAreas);
            } else {
                $this->setData(self::TEMPLATE_AREAS, []);
            }
        }

        return $this->getData(self::TEMPLATE_AREAS);
    }

    public function setTemplateAreas(array $areas): TemplateInterface
    {
        $this->setData(self::TEMPLATE_AREAS, $areas);

        return $this;
    }

    public function getTemplateAreasSerialized(): string
    {
        return (string)$this->getData(self::TEMPLATE_AREAS_SERIALIZED);
    }

    public function setTemplateAreasSerialized(string $dataSerialized): TemplateInterface
    {
        $this->setData(self::TEMPLATE_AREAS_SERIALIZED, $dataSerialized);

        return $this;
    }

    public function getThemeId(): ?int
    {
        return (int)$this->getData(ThemeInterface::ID);
    }

    public function setThemeId(int $themeId): TemplateInterface
    {
        $this->setData(ThemeInterface::ID, $themeId);

        return $this;
    }

    public function setTheme(ThemeInterface $theme): TemplateInterface
    {
        $this->theme = $theme;

        return $this;
    }

    public function getTheme(): ?ThemeInterface
    {
        if ($this->theme == null && $this->getThemeId()) {
            $this->theme = $this->themeFactory->create()
                ->load($this->getThemeId());
        }

        return $this->theme;
    }

    public function getAreaText(string $code): ?string
    {
        if (isset($this->getTemplateAreas()[$code])) {
            return (string)$this->getTemplateAreas()[$code];
        }

        return null;
    }

    public function setAreaText(string $code, string $content): TemplateInterface
    {
        $templateAreas = $this->getTemplateAreas();
        $templateAreas[$code] = $content;
        $this->setData(self::TEMPLATE_AREAS, $templateAreas);

        return $this;
    }

    public function getAreaCodeByContent(string $content): ?string
    {
        $code = null;
        foreach ($this->getTemplateAreas() as $areaCode => $areaContent) {
            if ($areaContent == $content) {
                $code = $areaCode;
                break;
            }
        }

        return $code;
    }

    public function getUpdatedAt(): string
    {
        return (string)$this->getData(self::UPDATED_AT);
    }

    public function setUpdatedAt(string $updatedAt): TemplateInterface
    {
        $this->setData(self::UPDATED_AT, $updatedAt);

        return $this;
    }

    public function getCreatedAt(): string
    {
        return (string)$this->getData(self::CREATED_AT);
    }

    public function setCreatedAt(string $createdAt): TemplateInterface
    {
        $this->setData(self::CREATED_AT, $createdAt);

        return $this;
    }

    public function isSystem(): bool
    {
        return $this->hasData(TemplateInterface::SYSTEM_ID) && $this->getData(TemplateInterface::SYSTEM_ID) > 0;
    }

    public function setSystemId(int $id): TemplateInterface
    {
        $this->setData(TemplateInterface::SYSTEM_ID, $id);

        return $this;
    }

    public function export(): string
    {
        $this->setData('theme', $this->getTheme()->getTitle());

        $path = $this->config->getTemplatePath() . '/' . $this->getTitle() . '.json';

        file_put_contents($path, $this->toJson());

        return $path;
    }

    public function import(string $filePath): TemplateInterface
    {
        $data = YamlParser::parse(file_get_contents($filePath));

        // retrieve system ID from fixture filename
        $pathParts = explode('/', $filePath);
        $nameParts = explode('_', array_pop($pathParts));
        $systemId  = array_shift($nameParts);

        /** @var Template $model */
        $model = $this->getCollection()
            ->addFieldToFilter('title', $data['title'])
            ->getFirstItem();

        $model->addData($data);

        /** @var Theme $theme */
        $theme = $this->themeFactory->create()->getCollection()
            ->addFieldToFilter('title', $data['theme'])
            ->getFirstItem();

        $model->setThemeId((int)$theme->getId());
        if (is_numeric($systemId)) {
            $model->setSystemId((int)$systemId);
        }

        $model->save();

        return $model;
    }
}
