<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.2
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Repository;

use Magento\Framework\EntityManager\EntityManager;
use Magento\Framework\Exception\NoSuchEntityException;
use Mirasvit\Email\Api\Data\QueueInterface;
use Mirasvit\Email\Api\Data\QueueInterfaceFactory;
use Mirasvit\Email\Model\ResourceModel\Queue\Collection;
use Mirasvit\Email\Model\ResourceModel\Queue\CollectionFactory;
use Mirasvit\Email\Helper\Serializer;

class QueueRepository
{
    /**
     * @var QueueInterface[]
     */
    private $queueRegistry = [];

    /**
     * @var QueueInterface[]
     */
    private $queueByHashRegistry = [];

    private $entityManager;

    private $serializer;

    private $queueFactory;

    private $collectionFactory;


    public function __construct(
        EntityManager         $entityManager,
        QueueInterfaceFactory $queueFactory,
        CollectionFactory     $collectionFactory,
        Serializer            $serializer
    ) {
        $this->queueFactory      = $queueFactory;
        $this->collectionFactory = $collectionFactory;
        $this->entityManager     = $entityManager;
        $this->serializer        = $serializer;
    }

    public function get(int $id): ?QueueInterface
    {
        if (isset($this->queueRegistry[$id])) {
            return $this->queueRegistry[$id];
        }

        $queue = $this->create();
        $queue = $this->entityManager->load($queue, $id);
        //$queue = $queue->load($id);

        if ($queue->getId()) {
            $this->queueRegistry[$id] = $queue;
        } else {
            return null;
        }

        return $queue;
    }

    public function getByUniqueHash(string $uniqueHash): QueueInterface
    {
        if (isset($this->queueByHashRegistry[$uniqueHash])) {
            return $this->queueByHashRegistry[$uniqueHash];
        }

        /** @var QueueInterface $queue */
        $queue = $this->getCollection()
            ->addFieldToFilter(QueueInterface::UNIQUE_HASH, $uniqueHash)
            ->getFirstItem();

        if ($queue->getId()) {
            $queue = $this->get((int)$queue->getId());
        }

        $this->queueByHashRegistry[$uniqueHash] = $queue;

        return $queue;
    }

    public function save(QueueInterface $queue): QueueInterface
    {
        if ($queue->hasData(QueueInterface::ARGS)) {
            $queue->setArgsSerialized($this->serializer->serialize($queue->getData(QueueInterface::ARGS)));
        }

        if (!$queue->hasData(QueueInterface::STATUS)) {
            $queue->setStatus(QueueInterface::STATUS_PENDING)
                ->setData(QueueInterface::STATUS_MESSAGE, (string)__('Default status'));
        }

        $this->prepareHistory($queue);

        if (!$queue->getUniqHash()) {
            $queue->setUniqHash(hash('sha256', $queue->getUniqKey()));
        }

        return $this->entityManager->save($queue);
    }

    public function delete(QueueInterface $queue): bool
    {
        return $this->entityManager->delete($queue);
    }

    public function getCollection(): Collection
    {
        return $this->collectionFactory->create();
    }

    public function create(): QueueInterface
    {
        return $this->queueFactory->create();
    }

    private function prepareHistory(QueueInterface $queue): void
    {
        if ($queue->getOrigData(QueueInterface::STATUS) != $queue->getData(QueueInterface::STATUS)) {
            $newStatus = $queue->getData(QueueInterface::STATUS);

            $historyMessage = (string)__("%1 - status changed to '%2'", date('M d, Y H:i:s'), $newStatus);
            if ($queue->hasData(QueueInterface::STATUS_MESSAGE)) {
                $historyMessage .= ' [' . $queue->getData(QueueInterface::STATUS_MESSAGE) . ']';
            }

            $history = $queue->getData(QueueInterface::HISTORY);

            $history .= $historyMessage . PHP_EOL;

            $queue->setData(QueueInterface::HISTORY, $history);
        }
    }
}
