<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.2
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\Email\Model\Trigger;

use Mirasvit\Email\Api\Data\ChainInterface;
use Mirasvit\Email\Api\Data\QueueInterface;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Model\QueueFactory;
use Mirasvit\Email\Model\ResourceModel\Queue\Collection as QueueCollection;
use Mirasvit\Email\Repository\QueueRepository;
use Mirasvit\Email\Service\EventManagement;
use Mirasvit\Event\Api\Data\EventInterface;
use Mirasvit\Event\Api\Repository\EventRepositoryInterface;

class Handler
{
    protected $eventRepository;

    protected $queueRepository;

    protected $queueFactory;

    private   $eventManagement;

    public function __construct(
        EventManagement          $eventManagement,
        EventRepositoryInterface $eventRepository,
        QueueRepository          $queueRepository,
        QueueFactory             $queueFactory
    ) {
        $this->eventRepository = $eventRepository;
        $this->queueRepository = $queueRepository;
        $this->queueFactory    = $queueFactory;
        $this->eventManagement = $eventManagement;
    }

    /**
     * @param TriggerInterface      $trigger
     * @param EventInterface[]|null $events
     *
     * @return $this
     */
    public function handleEvents(TriggerInterface $trigger, array $events = null): self
    {
        if (!$events) {
            $events = $this->eventRepository->getCollection()
                ->addFieldToFilter(EventInterface::IDENTIFIER, ['in' => $trigger->getEvents()])
                ->addFieldToFilter(EventInterface::KEY, ['nlike' => 'test_%']) // ignore test events
                ->setOrder(EventInterface::CREATED_AT, 'asc');

            $this->eventManagement->addNewFilter($events, $trigger->getId(), $trigger->getStoreIds());
        }

        /** @var EventInterface $event */
        foreach ($events as $event) {
            $this->handleEvent($trigger, $event);
        }

        return $this;
    }

    public function handleEvent(TriggerInterface $trigger, EventInterface $event): self
    {
        $this->processEvent($trigger, $event);
        $this->eventManagement->addProcessedTriggerId((int)$event->getId(), $trigger->getId());

        return $this;
    }

    /**
     * Trigger trigger by event
     */
    public function triggerEvent(TriggerInterface $trigger, EventInterface $event): bool
    {
        if (!$trigger->validateRules($event->getParams())) {
            return false;
        }

        /** @var \Mirasvit\Email\Model\Trigger\Chain $chain */
        foreach ($trigger->getChainCollection() as $chain) {
            $queue = $this->enqueue($trigger, $chain, $event);

            if (!$queue) {
                continue;
            }

            if ($event->getParam('force')) {
                $queue->send();
            }
        }

        return true;
    }

    /**
     * Add new message to queue.
     */
    public function enqueue(TriggerInterface $trigger, ChainInterface $chain, EventInterface $event): ?QueueInterface
    {
        $uniqueKey = "{$event->getKey()}|{$trigger->getId()}|{$chain->getId()}";
        $params    = $event->getParams();

        // Calculate scheduled at date starting from event registration date
        if (isset($params['is_test'])) {
            $scheduledAt = date('Y-m-d H:i:s', strtotime($params['created_at']));
        } else {
            $scheduledAt = date('Y-m-d H:i:s', $chain->getScheduledAt(strtotime($event->getCreatedAt())));
        }

        // If there are still queued messages - do not queue another one, ignore it
        $queueCollection = $this->getQueueCollection($trigger->getId(), $chain->getId(), $uniqueKey, $scheduledAt);
        if ($queueCollection->count() != 0) {
            return null;
        }

        // Enqueue message
        $recipientEmail = $params['customer_email'];
        $recipientName  = $params['customer_name'];
        if ($trigger->getIsAdmin() && !isset($params['force'])) {
            $recipientEmail = $trigger->getAdminEmail();
            $recipientName  = "Administrator";
        }

        if (!$recipientEmail) {
            return null;
        }

        $queue = $this->queueRepository->create();
        $queue->setTriggerId($trigger->getId())
            ->setChainId($chain->getId())
            ->setUniqKey($uniqueKey)
            ->setSenderEmail($trigger->getSenderEmail((int)$params['store_id']))
            ->setSenderName($trigger->getSenderName((int)$params['store_id']))
            ->setRecipientEmail($recipientEmail)
            ->setRecipientName($recipientName)
            ->setArgs($params)
            ->setScheduledAt($scheduledAt);

        $this->queueRepository->save($queue);

        return $queue;
    }

    /**
     * Handle one event
     */
    protected function processEvent(TriggerInterface $trigger, EventInterface $event): self
    {
        if (in_array($event->getIdentifier(), $trigger->getCancellationEvents())) {
            $this->cancelTrigger($trigger, $event);
        }

        if (in_array($event->getIdentifier(), $trigger->getTriggeringEvents())) {
            $this->triggerEvent($trigger, $event);
        }

        return $this;
    }

    /**
     * Cancel trigger by event
     */
    protected function cancelTrigger(TriggerInterface $trigger, EventInterface $event): self
    {
        $params = $event->getParams();

        $recipientEmail = $params['customer_email'];

        if ($trigger->getIsAdmin()) {
            $recipientEmail = $trigger->getAdminEmail();
        }

        foreach ($trigger->getChainCollection() as $chain) {
            $uniqueKey = "{$event->getKey()}|{$trigger->getId()}|{$chain->getId()}";

            $queueCollection = $this->queueRepository->getCollection();
            $queueCollection
                ->addFieldToFilter(QueueInterface::STATUS, ['neq' => QueueInterface::STATUS_SENT])
                ->addFieldToFilter(QueueInterface::TRIGGER_ID, $trigger->getId())
                ->addFieldToFilter(QueueInterface::RECIPIENT_EMAIL, $recipientEmail)
                ->addFieldToFilter(QueueInterface::UNIQUE_KEY, $uniqueKey);

            foreach ($queueCollection as $queue) {
                $queue->cancel((string)__('Cancellation Event (%1 - %2)', $event->getKey(), $event->getIdentifier()));
            }
        }

        return $this;
    }

    private function getQueueCollection(int $triggerId, int $chainId, string $uniqueKey, string $gmtScheduledAt): QueueCollection
    {
        $queueCollection = $this->queueRepository->getCollection();
        $queueCollection->addFieldToFilter(TriggerInterface::ID, $triggerId)
            ->addFieldToFilter(ChainInterface::ID, $chainId)
            ->addFieldToFilter(QueueInterface::UNIQUE_KEY, $uniqueKey)
            ->addFieldToFilter(QueueInterface::SCHEDULED_AT, $gmtScheduledAt);

        return $queueCollection;
    }
}
