<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.2
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Model\Queue;

use Magento\Framework\App\Area;
use Magento\Framework\App\ProductMetadataInterface as ProductMetadata;
use Magento\Framework\App\State as AppState;
use Magento\Framework\Mail\MessageFactory as MailMessageFactory;
use Magento\Framework\Mail\MessageInterface;
use Magento\Framework\Mail\TransportInterfaceFactory;
use Magento\Framework\Registry;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Framework\Validator\EmailAddress;
use Magento\Store\Model\App\Emulation;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Sales\Model\ResourceModel\Sale\CollectionFactory as SaleCollectionFactory;
use Magento\Framework\Module\Manager;
use Mirasvit\Email\Api\Data\QueueInterface;
use Mirasvit\Email\Controller\RegistryConstants;
use Mirasvit\Email\Helper\Data as Helper;
use Mirasvit\Email\Model\ConfigProvider;
use Mirasvit\Email\Model\Queue;
use Mirasvit\Email\Model\ResourceModel\Queue\CollectionFactory as QueueCollectionFactory;
use Mirasvit\Email\Model\Unsubscription;
use Mirasvit\Event\Event\Sales\OrderStatusEvent;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Sender
{
    protected $unsubscription;

    protected $queueCollectionFactory;

    protected $config;

    protected $mailMessageFactory;

    protected $appEmulation;

    protected $appState;

    protected $mailTransportFactory;

    protected $storeManager;

    protected $moduleManager;

    protected $helper;

    private $transportBuilder;

    private $registry;

    private $modifiers;

    private $productMetadata;

    private $emailValidator;

    private $date;

    private $saleCollectionFactory;

    public function __construct(
        Registry                  $registry,
        TransportBuilder          $transportBuilder,
        Unsubscription            $unsubscription,
        QueueCollectionFactory    $queueCollectionFactory,
        ConfigProvider            $config,
        DateTime                  $date,
        MailMessageFactory        $mailMessageFactory,
        Emulation                 $appEmulation,
        AppState                  $appState,
        TransportInterfaceFactory $mailTransportFactory,
        StoreManagerInterface     $storeManager,
        Manager                   $moduleManager,
        Helper                    $helper,
        ProductMetadata           $productMetadata,
        EmailAddress              $emailValidator,
        SaleCollectionFactory     $saleCollectionFactory,
        array $modifiers = []
    ) {
        $this->registry               = $registry;
        $this->unsubscription         = $unsubscription;
        $this->queueCollectionFactory = $queueCollectionFactory;
        $this->config                 = $config;
        $this->date                   = $date;
        $this->mailMessageFactory     = $mailMessageFactory;
        $this->appEmulation           = $appEmulation;
        $this->appState               = $appState;
        $this->mailTransportFactory   = $mailTransportFactory;
        $this->storeManager           = $storeManager;
        $this->moduleManager          = $moduleManager;
        $this->helper                 = $helper;
        $this->transportBuilder       = $transportBuilder;
        $this->modifiers              = $modifiers;
        $this->emailValidator         = $emailValidator;
        $this->productMetadata        = $productMetadata;
        $this->saleCollectionFactory  = $saleCollectionFactory;
    }

    public function send(QueueInterface $queue, bool $force = false): bool
    {
        if (!$this->canSend($queue) && !$force) {
            return false;
        }

        // register current email queue model instance
        $this->registry->register(RegistryConstants::CURRENT_QUEUE, $queue, true);

        $this->appEmulation->startEnvironmentEmulation((int)$queue->getArgs('store_id'), Area::AREA_FRONTEND, true);
        $subject = $queue->getMailSubject();
        $this->appEmulation->stopEnvironmentEmulation();

        $this->appEmulation->startEnvironmentEmulation((int)$queue->getArgs('store_id'), Area::AREA_FRONTEND, true);

        $body = $queue->getMailContent();

        foreach ($this->modifiers as $modifier) {
            $body = $modifier->modifyContent($queue, $body);
        }

        $body = $this->helper->prepareQueueContent($body, $queue);
        $this->appEmulation->stopEnvironmentEmulation();

        $this->appEmulation->startEnvironmentEmulation((int)$queue->getArgs('store_id'), Area::AREA_FRONTEND, true);

        $recipients = explode(',', $queue->getRecipientEmail());
        if ($this->config->isSandbox() && !$queue->getArg('force')) {
            $recipients = explode(',', $this->config->getSandboxEmail());
        }

        foreach ($recipients as $index => $email) {
            $name = $queue->getRecipientName();
            if (count($recipients) > 1) {
                $name .= ' - ' . ($index + 1);
            }
            unset($recipients[$index]);
            $recipients[$name] = $email;
        }

        //trim spaces and remove all empty items
        $copyTo = array_filter(array_map('trim', explode(',', (string)$queue->getTrigger()->getCopyEmail())));
        foreach ($copyTo as $bcc) {
            $this->transportBuilder->addBcc($bcc);
        }

        $this->transportBuilder
            ->setMessageType(MessageInterface::TYPE_HTML)
            ->setSubject($subject)
            ->setBody($body)
            ->setReplyTo($queue->getSenderEmail(), $queue->getSenderName());

        $magentoVersion = $this->productMetadata->getVersion();

        if ($this->moduleManager->isEnabled('Mageplaza_Smtp')) {
            /** @var mixed $builder */
            $builder = $this->transportBuilder;
            $builder->setFrom(['email' => $queue->getSenderEmail(), 'name' => $queue->getSenderName()]);
        } elseif (version_compare($magentoVersion, '2.3.0', '>=')) {
            /** @var mixed $builder */
            $builder = $this->transportBuilder;
            $builder->setFrom([$queue->getSenderEmail() => $queue->getSenderName()]);
        } else {
            $this->transportBuilder->setFrom($queue->getSenderEmail(), $queue->getSenderName());
        }

        foreach ($recipients as $name => $email) {
            $this->transportBuilder->addTo($email, $name);
        }

        $transport = $this->transportBuilder->getTransport();
        $transport->sendMessage();

        $queue->delivery();

        $this->appEmulation->stopEnvironmentEmulation();

        return true;
    }

    protected function canSend(QueueInterface $queue): bool
    {
        $args = $queue->getArgs();

        if ($queue->getArg('force')) {
            return true;
        }

        if (!$this->emailValidator->isValid(trim($queue->getRecipientEmail()))) {
            $queue->cancel((string)__('Canceled by invalid recipient email address format'));

            return false;
        }

        if (time() - strtotime($queue->getScheduledAt()) > 2 * 24 * 60 * 60) {
            $queue->miss((string)__('Scheduled at %1, attempt to send after 2 days', $queue->getScheduledAt()));

            return false;
        }

        // check unsubscription
        if ($this->unsubscription->isUnsubscribed($queue->getRecipientEmail(), $queue->getTriggerId())) {
            $queue->unsubscribe((string)__('Customer %1 is unsubscribed', $queue->getRecipientEmail()));

            return false;
        }

        // check rules
        if (!$queue->getTrigger()->validateRules($args)) {
            $queue->cancel((string)__('Canceled by trigger rules'));

            return false;
        }

        // check limitation
        if (!$this->isValidByLimit($args)) {
            $queue->cancel((string)__('Canceled by global limitation settings'));

            return false;
        }

        if (!$queue->getTemplate()) {
            $queue->cancel((string)__('Missed Template'));

            return false;
        }

        // check trigger Cancellation event rules for abandoned carts
        if ($queue->getTrigger()->getEvent() == 'quote_abandoned') {
            $cancellationEvents = $queue->getTrigger()->getCancellationEvents();

            if (!empty($cancellationEvents)) {
                $quoteId = (string)$queue->getArg('quote_id');

                $saleCollection = $this->saleCollectionFactory->create();
                $order          = $saleCollection->addFieldToFilter('quote_id', $quoteId);

                if (!empty($order->getData())) {
                    $orderStatus = $order->getData()[0]['status'];

                    if (in_array(OrderStatusEvent::IDENTIFIER . $orderStatus, $cancellationEvents)) {
                        $queue->cancel((string)__('Canceled by trigger Сancellation event rules'));

                        return false;
                    }
                }
            }
        }

        return true;
    }

    protected function isValidByLimit(array $args): bool
    {
        $result     = true;
        $emailLimit = $this->config->getEmailLimit();
        $hourLimit  = $this->config->getEmailLimitPeriod() * 60 * 60;
        if (in_array(0, [$emailLimit, $hourLimit])) {
            return $result;
        }

        $gmtTimestampMinusLimit = $this->date->timestamp() - $hourLimit;
        $filterDateFrom         = $this->date->gmtDate(null, $gmtTimestampMinusLimit);

        $queues = $this->queueCollectionFactory->create()
            ->addFieldToFilter('recipient_email', $args['customer_email'])
            ->addFieldToFilter('status', QueueInterface::STATUS_SENT)
            ->addFieldToFilter('updated_at', ['gt' => $filterDateFrom]);

        if ($queues->count() >= $emailLimit) {
            $result = false;
        }

        return $result;
    }
}
