<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.2
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Controller\Action;

use Magento\Framework\Controller\ResultFactory;
use Mirasvit\Email\Api\Data\ChainInterface;
use Mirasvit\Email\Repository\Trigger\ChainRepository;
use Mirasvit\Email\Repository\TriggerRepository;
use Mirasvit\Email\Service\TestSender;
use Mirasvit\Email\Controller\Action;
use Mirasvit\Email\Service\FrontSessionInitiator;

class Send extends Action
{
    private $chainRepository;

    private $triggerRepository;

    private $emailSessionManager;

    private $testSender;


    public function __construct(
        TestSender $testSender,
        FrontSessionInitiator $emailSessionManager,
        ChainRepository $chainRepository,
        TriggerRepository $triggerRepository,
        Context $context
    ) {
        parent::__construct($context);

        $this->testSender          = $testSender;
        $this->chainRepository     = $chainRepository;
        $this->triggerRepository   = $triggerRepository;
        $this->emailSessionManager = $emailSessionManager;
    }

    /**
     * {@inheritdoc}
     */
    public function execute()
    {
        /** @var \Magento\Framework\Controller\Result\Json $resultPage */
        $resultPage = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        if (!$this->getRequest()->isAjax()) {
            return $resultPage->setData(['success' => false, 'message' =>(string)__('Operation is not allowed')]);
        }

        if (!$this->getRequest()->getParam('email')) {
            $resultPage->setData(['success' => false, 'message' =>(string)__('Please specify an email address')]);

            return $resultPage;
        }

        $chain   = $this->chainRepository->get((int)$this->getRequest()->getParam(ChainInterface::ID));
        $trigger = $this->triggerRepository->get($chain->getTriggerId());

        if (!$trigger->getEvent()) {
            $resultPage->setData(['success' => false, 'message' =>(string)__('Please specify trigger event first')]);

            return $resultPage;
        }

        if ($chain->getId()) {
            try {
                $this->testSender->sendChain($chain, $this->getRequest()->getParam('email'));
            } catch (\Exception $e) {
                return $resultPage->setData(['success' => false, 'message' => $e->getMessage()]);
            }

            return $resultPage->setData(['success' => true, 'message' =>(string)__('Test email was successfully sent')]);
        }

        $resultPage->setData(['success' => false, 'message' =>(string)__('Unable to find email to send')]);

        return $resultPage;
    }
}
