<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.4.1
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\EmailDesigner\Service;

use Magento\Framework\DataObject;
use Mirasvit\EmailDesigner\Service\TemplateEngine\Liquid\Variable;

class VariableResolver
{
    /**
     * Docblock tag used as the variable name.
     */
    const DOCBLOCK_TAG_DESCRIPTION = 'desc';

    /**
     * Docblock tag used as the variable namespace.
     */
    const DOCBLOCK_TAG_NAMESPACE = 'namespace';

    /**
     * Docblock tag used as the variable filter.
     */
    const DOCBLOCK_TAG_FILTER = 'filter';

    /**
     * Docblock tag used to highlight a return type.
     */
    const DOCBLOCK_TAG_RETURN = 'return';

    /**
     * Key used to identify callback.
     */
    const CALLBACK = 'callback';

    /**
     * List of variable hosts.
     */
    protected $variables = [];

    /**
     * @var DataObject
     */
    protected $context;

    /**
     * List of variable hosts.
     *
     * @var Variable\AbstractVariable[]
     */
    private $origVariables;


    public function __construct(array $variables = [])
    {
        $this->origVariables = $variables;
        $this->variables     = $variables;
    }

    public function resolve(string $name, array $args = [])
    {
        $result = false;
        $methodName = 'get' . str_replace(' ', '', ucwords(str_replace('_', ' ', $name)));

        foreach ($this->getVariables(true) as $variable) {
            if ($variable instanceof Variable\AbstractVariable) { // set variable context
                $variable->setContext($this->context);
            }

            // invoke method with given name
            if ($this->canInvoke($variable, $name, $args)) {
                return $this->invoke($variable, $name, $args);
            }

            // invoke method name prepended with the "get" keyword
            if ($this->canInvoke($variable, $methodName, $args)) {
                return $this->invoke($variable, $methodName, $args);
            }

            // check existence of a data with key $name in the data object
            if ($variable instanceof DataObject && $variable->hasData($name)) {
                return $variable->getData($name);
            }

            // invoke object itself
            /*if ($variable instanceof Variable\AbstractVariable && $name == $variable->getNamespace()) {
                return $variable();
            }*/
        }

        return $result;
    }

    /**
     * We cannot invoke $variable's method if number of passed $args less than number of required args.
     *
     * @param object $variable - possible host object of invoked method
     * @param string $name     - method name
     * @param array  $args     - method arguments
     *
     * @return bool
     */
    private function canInvoke($variable, $name, array $args = [])
    {
        if (!method_exists((object)$variable, $name)) {
            return false;
        }

        $reflectionMethod = new \ReflectionMethod($variable, $name);
        if (count($args) < $reflectionMethod->getNumberOfRequiredParameters()
            || count($args) > $reflectionMethod->getNumberOfParameters()
        ) {
            return false;
        }

        return true;
    }

    public function getVariables(bool $isAll = false): array
    {
        foreach ($this->variables as $key => $variable) {
            // if a variable is a callback - then execute the callback and store its result as a variable
            // where lazy loading actually happens
            if (is_array($variable) && is_callable($variable)) {
                $result = call_user_func($variable);
                if (is_object($result)) {
                    $this->variables[get_class($result)] = $result;
                } else {
                    unset($this->variables[$key]);
                }
            }
        }

        if ($isAll) {
            return $this->variables;
        }

        $variables = [];
        foreach ($this->variables as $variable) {
            if ($variable instanceof Variable\AbstractVariable) {
                $variables[$variable->getNamespace()] = $variable;
            }
        }

        return $variables;
    }

    public function getOrigVariables(): array
    {
        return $this->origVariables;
    }

    /**
     * @param object $object
     *
     * @return array
     */
    public function getVariablesFor($object)
    {
        $variables = [];
        foreach ($this->origVariables as $variable) {
            if ($variable->isFor($object)) {
                $variables[] = $variable;
            }
        }

        return $variables;
    }

    /**
     * @param mixed $variable
     *
     * @return $this
     */
    public function addVariable($variable)
    {
        if (!is_object($variable) && !(is_array($variable) && isset($variable[Variable\AbstractVariable::CALLBACK]))) {
            throw new \InvalidArgumentException('"variable" argument should be an object or callable.');
        }

        if (is_array($variable) && array_key_exists(Variable\AbstractVariable::CALLBACK, $variable)) {
            $class    = $variable[Variable\AbstractVariable::DOCBLOCK_TAG_RETURN];
            $variable = $variable[Variable\AbstractVariable::CALLBACK];
        }

        // add new variable to the beginning
        array_unshift($this->variables, $variable);

        if (isset($class)) {
            $variable = $class;
        }

        // prepend variables associated with the newly added variable
        foreach ($this->getVariablesFor($variable) as $variableFor) {
            $this->addVariable($variableFor);
        }

        return $this;
    }

    public function getVariable(string $namespace): ?Variable\AbstractVariable
    {
        foreach ($this->origVariables as $variable) {
            if ($variable->getNamespace() === $namespace) {
                return $variable;
            }
        }

        return null;
    }

    public function setContext(DataObject $context): self
    {
        $this->context = $context;

        return $this;
    }

    /**
     * Reset variables on cloning.
     */
    public function __clone()
    {
        $this->variables = [];
    }

    /**
     * Invoke method on a variable with given args.
     * @param mixed $variable
     * @param string $methodName
     * @param array $args
     * @return mixed|string
     * @return mixed|string
     */
    private function invoke($variable, $methodName, $args)
    {
        $result = call_user_func_array([$variable, $methodName], $args); // maybe remove args

        if ($result instanceof \Magento\Framework\Phrase) {
            $result = $result->__toString();
        }

        return $result;
    }
}
