<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.4.1
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Block\Adminhtml\Event;

use Magento\Backend\Block\Widget\Context;
use Magento\Backend\Block\Widget\Grid\Extended as ExtendedGrid;
use Magento\Backend\Helper\Data as BackendHelper;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Repository\TriggerRepository;
use Mirasvit\Event\Ui\Event\Source\Event as SourceEvent;
use Mirasvit\Event\Api\Data\EventInterface;
use Mirasvit\Event\Api\Repository\EventRepositoryInterface;

class Grid extends ExtendedGrid
{
    protected $eventRepository;

    protected $sourceEvent;

    protected $context;

    private $triggerRepository;

    public function __construct(
        TriggerRepository $triggerRepository,
        EventRepositoryInterface $eventRepository,
        SourceEvent $sourceEvent,
        Context $context,
        BackendHelper $backendHelper
    ) {
        $this->triggerRepository = $triggerRepository;
        $this->eventRepository = $eventRepository;
        $this->sourceEvent = $sourceEvent;
        $this->context = $context;

        parent::__construct($context, $backendHelper);
    }

    /**
     * {@inheritdoc}
     */
    protected function _construct()
    {
        parent::_construct();
        $this->setId('email_event_grid');
        $this->setDefaultSort('created_at');
        $this->setDefaultDir('DESC');
        $this->setSaveParametersInSession(true);
    }

    /**
     * {@inheritdoc}
     */
    protected function _prepareCollection()
    {
        $collection = $this->eventRepository->getCollection();

        $this->setCollection($collection);

        return parent::_prepareCollection();
    }

    /**
     * {@inheritdoc}
     */
    protected function _prepareColumns()
    {
        $this->addColumn(EventInterface::ID, [
            'header' => (string)__('ID'),
            'type'   => 'number',
            'index'  => EventInterface::ID,
        ]);

        $this->addColumn(EventInterface::IDENTIFIER, [
            'header'  => (string)__('Event'),
            'index'   => EventInterface::IDENTIFIER,
            'type'    => 'options',
            'options' => $this->sourceEvent->toHash()
        ]);

        $this->addColumn('message', [
            'header'   => (string)__('Message'),
            'index'    => EventInterface::PARAMS_SERIALIZED,
            'filter'   => false,
            'sortable' => false,
            'renderer' => 'Mirasvit\Email\Block\Adminhtml\Event\Grid\Renderer\Message',
        ]);

        $this->addColumn(EventInterface::PARAMS_SERIALIZED, [
            'header'   => (string)__('Arguments'),
            'index'    => EventInterface::PARAMS_SERIALIZED,
            'filter'   => false,
            'sortable' => false,
            'renderer' => 'Mirasvit\Email\Block\Adminhtml\Event\Grid\Renderer\Args',
        ]);

        $this->addColumn(EventInterface::CREATED_AT, [
            'header' => (string)__('Created At'),
            'index'  => EventInterface::CREATED_AT,
            'type'   => 'datetime',
        ]);

        $this->addColumn('triggers', [
            'header'   => (string)__('Triggers'),
            'index'    => 'triggers',
            'renderer' => 'Mirasvit\Email\Block\Adminhtml\Event\Grid\Renderer\Triggers',
            'filter'   => false,
            'sortable' => false,
        ]);

        if (!$this->context->getStoreManager()->isSingleStoreMode()) {
            $this->addColumn('store_ids', [
                'header'     => (string)__('Store'),
                'index'      => 'store_id',
                'type'       => 'store',
                'store_all'  => true,
                'store_view' => true,
                'sortable'   => false
            ]);
        }

        $this->addColumn('action', [
            'header'    => (string)__('Actions'),
            'width'     => '100',
            'type'      => 'action',
            'getter'    => 'getId',
            'actions'   => [
                [
                    'caption' => (string)__('Reset & Process'),
                    'url'     => ['base' => '*/*/reset'],
                    'field'   => 'id',
                ],
                [
                    'caption' => (string)__('Delete'),
                    'url'     => ['base' => '*/*/delete'],
                    'field'   => 'id',
                ],
            ],
            'filter'    => false,
            'sortable'  => false,
            'is_system' => true,
        ]);

        return parent::_prepareColumns();
    }

    /**
     * @return $this|ExtendedGrid
     */
    protected function _prepareMassaction()
    {
        $this->setMassactionIdField(EventInterface::ID);
        $this->getMassactionBlock()->setFormFieldName(EventInterface::ID);

        $this->getMassactionBlock()->addItem('delete', [
            'label'    => (string)__('Delete'),
            'url'      => $this->getUrl('*/*/massDelete'),
            'confirm'  => (string)__('Are you sure?')
        ]);

        $triggers = $this->triggerRepository->getCollection()
            ->addFieldToFilter(TriggerInterface::RULE_SERIALIZED, ['neq' => '[]'])
            ->toOptionArray();

        $this->getMassactionBlock()->addItem('validate', [
            'label' => (string)__('Validate'),
            'url'   => $this->getUrl('*/*/massValidate', ['_current' => true]),
            'additional' => [
                'trigger' => [
                    'name'   => TriggerInterface::ID,
                    'type'   => 'select',
                    'class'  => 'required-entry',
                    'label'  => (string)__('Select Trigger'),
                    'values' => $triggers
                ]
            ]
        ]);

        $this->getMassactionBlock()->addItem('reset', [
            'label'    => (string)__('Reset & Process'),
            'url'      => $this->getUrl('*/*/massReset'),
            'confirm'  => (string)__('Are you sure?')
        ]);

        return $this;
    }

    /**
     * {@inheritdoc}
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function getRowUrl($row)
    {
        return false;
    }
}
