<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.4.3
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\EmailDesigner\Service\TemplateEngine;


use Magento\Framework\ObjectManagerInterface;
use Mirasvit\EmailDesigner\Service\TemplateEngine;
use Mirasvit\EmailDesigner\Service\TemplateEngine\Liquid\Variable\Context;
use Mirasvit\EmailDesigner\Service\VariableResolver;
use Mirasvit\EmailDesigner\Service\TemplateEngine\Liquid\DropProxy;
use Liquid\Template as LiquidTemplate;
use Liquid\Liquid as LiquidLib;

class Liquid
{
    /**
     * Ignore directives used by magento template engine to avoid conflict.
     *
     * @var string[]
     */
    private $ignoredDirectives = [
        'template',
        'depend',
        '\/depend',
        'var',
        'if',
        '\/if',
        'block',
        'layout',
        'view',
        'media',
        'store',
        'trans',
        'protocol',
        'config',
        'customvar',
        'css',
        'inlinecss',
    ];

    private $filters = [];

    /**
     * @var object[]
     */
    private $filterObjects = [];

    private $objectManager;

    private $context;

    private $variableResolver;

    public function __construct(
        Context $context,
        VariableResolver $variableResolver,
        ObjectManagerInterface $objectManager,
        array $filters = []
    ) {
        $this->variableResolver = $variableResolver;
        $this->context = $context;
        $this->filters = $filters;
        $this->objectManager = $objectManager;
    }

    public function render(string $template, array $variables = []): string
    {
        $this->context->unsetData();
        $this->context->addData($variables);

        // create drop variables for liquid
        $variables['this'] = new DropProxy($this->context, $this->variableResolver);
        foreach ($this->variableResolver->getOrigVariables() as $variable) {
            $variable->setContext($this->context);
            $variableResolver = clone $this->variableResolver;
            if ($variable->getCallback()) { // implement lazy loading
                $variableResolver->addVariable($variable->getCallback());
            } else {
                $variableResolver->addVariable($variable);
            }

            $variables[$variable->getNamespace()] = new DropProxy($this->context, $variableResolver);
        }

        // change variable start regexp to exclude Magento Template Engine directives
        $variableStartRegexp = LiquidLib::get('VARIABLE_START') . '(?!' . implode('|', $this->ignoredDirectives) . ')';
        LiquidLib::set('VARIABLE_START', $variableStartRegexp);

        // process template
        $liquidTemplate = new LiquidTemplate();
        $liquidTemplate->parse($template);

        $result = $liquidTemplate->render($variables, $this->getFilters(), $this->context->getData());

        return $result;
    }

    /**
     * Get filter instances.
     *
     * @return object[]
     */
    private function getFilters(): array
    {
        if (empty($this->filterObjects)) {
            foreach ($this->filters as $filter) {
                $this->filterObjects[] = $this->objectManager->create($filter);
            }
        }

        return $this->filterObjects;
    }
}
