<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.4.3
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\EmailDesigner\Helper;


use Magento\Framework\DataObject;

class ReflectionDocblockHelper extends DataObject
{
    const NAME          = 'name';
    const TYPE          = 'type';
    const DESCRIPTION   = 'description';
    const VARIABLE_NAME = 'variableName';

    private $tags             = [];

    private $cleanDocComment  = '';

    private $shortDescription = '';

    private $longDescription  = '';

    /**
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    public function __construct(string $docComment = '')
    {
        if (!$docComment) {
            return;
        }

        // First remove doc block line starters
        $docComment = preg_replace('#[ \t]*(?:\/\*\*|\*\/|\*)?[ ]{0,1}(.*)?#', '$1', $docComment);
        $docComment = ltrim($docComment, "\r\n");

        $this->cleanDocComment = $docComment;

        // Next parse out the tags and descriptions
        $parsedDocComment = $docComment;
        $lineNumber       = $firstBlandLineEncountered = 0;

        while (($newlinePos = strpos($parsedDocComment, "\n")) !== false) {
            $lineNumber++;

            $line    = substr($parsedDocComment, 0, $newlinePos);
            $matches = array();

            if ((strpos($line, '@') === 0) && (preg_match('#^(@\w+.*?)(\n)(?:@|\r?\n|$)#s', $parsedDocComment, $matches))) {
                if (preg_match('#^@(return)\s+([^\s]+)(?:\s+(.*))?#', $matches[1], $m)) {
                    $this->tags['return'] = new DataObject([
                        self::NAME        => 'return',
                        self::TYPE        => $m[2],
                        self::DESCRIPTION => $m[3] ?? null
                    ]);
                } elseif(preg_match('#^@(param)\s+([^\s]+)(?:\s+(\$\S+))?(?:\s+(.*))?#s', $matches[1], $m)) {
                    $this->tags['param'][] = new DataObject([
                        self::NAME          => 'param',
                        self::TYPE          => $m[2],
                        self::VARIABLE_NAME => $m[3] ?? null,
                        self::DESCRIPTION   => $m[4] ?? null
                    ]);
                } elseif(preg_match('#^@(\w+)(?:\s+([^\s].*)|$)?#', $matches[1], $m)) {
                    $this->tags[$m[1]] = new DataObject([
                        self::NAME        => $m[1],
                        self::DESCRIPTION => $m[2] ?? null
                    ]);
                }

                $parsedDocComment = str_replace($matches[1] . $matches[2], '', $parsedDocComment);
            } else {
                if ($lineNumber < 3 && !$firstBlandLineEncountered) {
                    $this->shortDescription .= $line . "\n";
                } else {
                    $this->longDescription .= $line . "\n";
                }

                if ($line == '') {
                    $firstBlandLineEncountered = true;
                }

                $parsedDocComment = substr($parsedDocComment, $newlinePos + 1);
            }

        }

        $this->shortDescription = rtrim($this->shortDescription);
        $this->longDescription  = rtrim($this->longDescription);
    }

    public function getTags(): array
    {
        return $this->tags;
    }

    public function hasTag(string $tagName): bool
    {
        return isset($this->tags[$tagName]);
    }

    /**
     * @param string $tagName
     *
     * @return DataObject|DataObject[]|null
     */
    public function getTag(string $tagName)
    {
        return $this->hasTag($tagName) ? $this->tags[$tagName] : null;
    }

    public function getShortDescription(): string
    {
        return $this->shortDescription;
    }

    public function getLongDescription(): string
    {
        return $this->longDescription;
    }

    public function getCleanDocComment(): string
    {
        return $this->cleanDocComment;
    }
}
