<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.4.3
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Repository;

use Magento\Framework\EntityManager\EntityManager;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Api\Data\TriggerInterfaceFactory;
use Mirasvit\Email\Model\ResourceModel\Trigger\Collection;
use Mirasvit\Email\Model\ResourceModel\Trigger\CollectionFactory;
use Mirasvit\Core\Service\SerializeService;

class TriggerRepository
{
    /**
     * @var TriggerInterface[]
     */
    private $triggerRegistry = [];

    private $entityManager;

    private $modelFactory;

    private $collectionFactory;


    public function __construct(
        EntityManager $entityManager,
        TriggerInterfaceFactory $modelFactory,
        CollectionFactory $collectionFactory
    ) {
        $this->entityManager     = $entityManager;
        $this->modelFactory      = $modelFactory;
        $this->collectionFactory = $collectionFactory;
    }

    public function getCollection(): Collection
    {
        return $this->collectionFactory->create();
    }

    public function create(): TriggerInterface
    {
        return $this->modelFactory->create();
    }

    public function get(int $id): ?TriggerInterface
    {
        if (isset($this->triggerRegistry[$id])) {
            return $this->triggerRegistry[$id];
        }

        $trigger = $this->create();
        $trigger = $this->entityManager->load($trigger, $id);

        if ($trigger->getId()) {
            $this->triggerRegistry[$id] = $trigger;
        } else {
            return null;
        }

        return $trigger;
    }

    public function save(TriggerInterface $model): TriggerInterface
    {
        $model->setRuleSerialized(SerializeService::encode($model->getRule() ? $model->getRule() : []));
        $model->setStoreIds($model->getStoreIds());
        $model->setCancellationEvent($model->getCancellationEvent());
        $model->setAdminEmail($model->getAdminEmail());

        $this->entityManager->save($model);

        return $model;
    }

    public function delete(TriggerInterface $model): bool
    {
        return $this->entityManager->delete($model);
    }
}
