<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.5
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Service;

use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Model\Trigger\Handler as TriggerHandler;
use Mirasvit\Email\Repository\TriggerRepository;
use Mirasvit\Event\Api\Repository\EventRepositoryInterface;

class EventProcessor
{
    private $eventRepository;

    private $triggerRepository;

    private $triggerHandler;

    public function __construct(
        TriggerHandler           $triggerHandler,
        TriggerRepository        $triggerRepository,
        EventRepositoryInterface $eventRepository
    ) {
        $this->eventRepository   = $eventRepository;
        $this->triggerRepository = $triggerRepository;
        $this->triggerHandler    = $triggerHandler;
    }

    /**
     * {@inheritdoc}
     */
    public function execute(int $eventId)
    {
        $event = $this->eventRepository->get($eventId);
        if (!$event) {
            return $this;
        }

        foreach ($this->getTriggers($event->getIdentifier(), $event->getStoreId()) as $trigger) {
            $this->triggerHandler->handleEvent($trigger, $event);
        }
    }

    /**
     * Get collection of triggers that can process the event.
     *
     * @param string $eventCode
     * @param int    $storeId
     *
     * @return \Mirasvit\Email\Api\Data\TriggerInterface[]|\Mirasvit\Email\Model\ResourceModel\Trigger\Collection
     */
    private function getTriggers($eventCode, $storeId)
    {
        $triggers = $this->triggerRepository->getCollection();
        $triggers->addActiveFilter();
        $triggers->addFieldToFilter(
            [TriggerInterface::EVENT, TriggerInterface::CANCELLATION_EVENT],
            [
                ['eq' => $eventCode],
                ['finset' => $eventCode],
            ]
        );

        // filter by stores
        foreach ($triggers as $trigger) {
            $storeIds = $trigger->getStoreIds();
            if (count($storeIds)
                && !in_array($storeId, $storeIds)
                && !in_array(0, $storeIds)
            ) {
                $triggers->removeItemByKey($trigger->getId());
            }
        }

        return $triggers;
    }
}
