<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.3.1
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\EmailDesigner\Service\TemplateEngine\Php\Variable;

use Magento\SalesRule\Api\Data\CouponInterface;
use Magento\SalesRule\Model\CouponFactory;
use Magento\SalesRule\Model\Coupon\Massgenerator;
use Magento\SalesRule\Model\RuleFactory;
use Magento\SalesRule\Model\Rule;
use Mirasvit\Email\Api\Data\QueueInterface;
use Mirasvit\Email\Api\Data\ChainInterface;
use Mirasvit\EmailDesigner\Service\TemplateEngine\Php\Variable\Context;
use Mirasvit\Email\Model\ConfigProvider;

class Coupon
{
    protected $coupons = [];

    protected $config;

    private $context;

    private $ruleFactory;

    private $couponMassgenerator;

    private $couponFactory;

    public function __construct(
        CouponFactory $couponFactory,
        Massgenerator $couponMassgenerator,
        RuleFactory $ruleFactory,
        Context $context,
        ConfigProvider $config
    ) {
        $this->couponFactory       = $couponFactory;
        $this->couponMassgenerator = $couponMassgenerator;
        $this->ruleFactory         = $ruleFactory;
        $this->context             = $context;
        $this->config              = $config;
    }

    public function getCoupon(): CouponInterface
    {
        if ($this->context->getData('preview')) {
            // in preview mode, we create fake coupon
            $expirationDate = time() + rand(1, 30) * 24 * 60 * 60;

            $coupon = $this->couponFactory->create();
            $coupon->setCode('EML#####')
                ->setExpirationDate(date(\DateTime::ISO8601, $expirationDate))
                ->setType(1);

            return $coupon;
        } elseif ($this->context->getData('chain')) {
            /** @var \Mirasvit\Email\Model\Trigger\Chain $chain */
            $chain = $this->context->getData('chain');

            # if we already generated coupon for this chain
            if (isset($this->coupons[$this->getCouponKey($chain, $this->context->getData('queue'))])) {
                return $this->coupons[$this->getCouponKey($chain, $this->context->getData('queue'))];
            }

            if ($chain->getCouponEnabled()) {
                $rule = $this->ruleFactory->create()->load($chain->getCouponSalesRuleId());

                if ($rule->getId()) {
                    if ($rule->getUseAutoGeneration()) {
                        $coupon = $this->generateCoupon($rule, $chain);
                    } else {
                        $coupon = $this->couponFactory->create();
                        $coupon->setRule($rule)
                            ->setCode($rule->getCouponCode());
                    }

                    $this->coupons[$this->getCouponKey($chain, $this->context->getData('queue'))] = $coupon;

                    return $coupon;
                }
            }
        }

        return false;
    }

    protected function generateCoupon(Rule $rule, ChainInterface $chain): CouponInterface
    {
        $generator = $this->couponMassgenerator;

        $generator->addData([
            'length' => $this->config->getCouponLength(),
            'prefix' => $this->config->getCouponPrefix(),
            'suffix' => $this->config->getCouponSuffix(),
            'dash'   => $this->config->getCouponDash(),
        ]);

        $code   = $generator->generateCode();
        $coupon = $this->couponFactory->create();

        if ($chain->getCouponExpiresDays()) {
            $expirationDate = time() + $chain->getCouponExpiresDays() * 24 * 60 * 60;
            $coupon->setExpirationDate(date(\DateTime::ISO8601, $expirationDate));
        }

        $coupon->setRule($rule)
            ->setCode($code)
            ->setIsPrimary(false)
            ->setUsageLimit(1)
            ->setUsagePerCustomer(1)
            ->setType(1)
            ->setCreatedAt(date('Y-m-d H:i:s'))
            ->save();

        return $coupon;
    }

    private function getCouponKey(ChainInterface $chain, QueueInterface $queue): string
    {
        return $chain->getId() . '_' . $queue->getId();
    }
}
