<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.3.5
 * @copyright Copyright (C) 2022 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\EmailDesigner\Service\TemplateEngine\Liquid;


use Mirasvit\EmailDesigner\Service\TemplateEngine\Liquid\Variable;
use Mirasvit\EmailDesigner\Service\VariableResolver;
use Mirasvit\EmailDesigner\Helper\ReflectionDocblockHelper;

class VariableProvider
{
    private $variableResolver;


    public function __construct(
        VariableResolver $variableResolver
    ) {
        $this->variableResolver = $variableResolver;
    }

    /**
     * {@inheritdoc}
     */
    public function getVariables()
    {
        $variables = $this->variableResolver->getOrigVariables();
        $result = [
            [
                'label' => (string)__('General Variables'),
                'value' => [
                    [
                        'label' => (string)__('Customer Name'),
                        'value' => (string)__('{{ customer_name }}')
                    ],
                    [
                        'label' => (string)__('Customer Email'),
                        'value' => (string)__('{{ customer_email }}')
                    ],
                    [
                        'label' => (string)__('Customer First Name'),
                        'value' => (string)__('{{ customer_name | split: " " | first }}')
                    ],
                ]
            ]
        ];

        foreach ($variables as $variable) {
            $variableReflection = new \ReflectionClass($variable);
            /** @var \ReflectionMethod[] $reflectionMethods */
            $reflectionMethods = $variableReflection->getMethods(\ReflectionMethod::IS_PUBLIC);

            if ($methods = $this->collectMethods($variable, $reflectionMethods)) {
                $result[] = [
                    'label' => (string)__('%1 Variables', ucfirst((string)$variable->getVariableName())),
                    'value' => $methods
                ];
            }
        }

        return $result;
    }

    /**
     * Collect all available methods for user.
     *
     * @param Variable\AbstractVariable         $variable
     * @param \ReflectionMethod[] $reflectionMethods
     *
     * @return array
     */
    private function collectMethods(Variable\AbstractVariable $variable, array $reflectionMethods = [])
    {
        $methods = [];
        foreach ($reflectionMethods as $method) {
            if ($this->canUseMethod($method, $variable)) {
                $docblock = new ReflectionDocblockHelper($method->getDocComment());
                $methodName = $this->convertToLiquid($method->getName());

                // add methods available in the model object
                if ($this->isObjectReturned($docblock) && $variable->getNamespace() == $methodName) {
                    $returnType = $docblock->getTag('return')->getType();
                    $variableReflection = new \ReflectionClass($returnType);
                    $reflectionMethods = $variableReflection->getMethods(\ReflectionMethod::IS_PUBLIC);

                    $methods = array_merge(
                        $methods,
                        $this->collectMethods($variable, $reflectionMethods)
                    );

                    continue;
                }

                if ($definition = $this->getMethodDefinition($variable, $method)) {
                    array_unshift($methods, $definition);
                }
            }
        }

        return $methods;
    }

    /**
     * Determine whether the method can be used or not.
     *
     * @param \ReflectionMethod $method
     * @param Variable\AbstractVariable       $variable
     *
     * @return bool
     */
    private function canUseMethod(\ReflectionMethod $method, Variable\AbstractVariable $variable)
    {
        $canUse = false;
        if (strpos($method->getName(), 'get') === 0 // only getters
            && strpos($method->getDeclaringClass()->getName(), 'AbstractVariable') === false // ignore AbstractClass
            && $method->getDocComment() // skip methods without docblock
            && strpos($method->getDocComment(), '@Suppress') === false // tags without space throw an error
            && $method->getNumberOfRequiredParameters() === 0 // only methods without parameters
            && stripos($method->getDocComment(), '@inheritdoc') === false // ignore methods with "inheritdoc"
        ) {
            $canUse = true;
        }

        return $canUse;
    }

    /**
     * Convert method name to liquid compatible syntax.
     *
     * e.g., "getCouponCode" => coupon_code
     *
     * @param string $methodName
     *
     * @return string
     */
    private function convertToLiquid($methodName)
    {
        $methodWords = preg_split('/(?=[A-Z])/', $methodName); // split name by uppercase letters
        $methodWords = array_filter($methodWords, function($name) { // remove keyword "get" from words
            return $name !== 'get';
        });

        // join remained words with underscore
        return implode('_', array_map('strtolower', $methodWords));
    }

    /**
     * Whether the method return type is object or not.
     *
     * @param ReflectionDocblockHelper $docblock
     *
     * @return bool
     *
     */
    private function isObjectReturned(ReflectionDocblockHelper $docblock)
    {
        if ($docblock->hasTag('return')) {
            return strpos($docblock->getTag('return')->getType(), '\\') !== false;
        }

        return false;
    }

    /**
     * Get liquid variable definition.
     *
     * @param Variable\AbstractVariable $variable
     * @param \ReflectionMethod $method
     *
     * @return array
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    private function getMethodDefinition(Variable\AbstractVariable $variable, \ReflectionMethod $method)
    {
        $definition = [];
        $docblock   = new ReflectionDocblockHelper($method->getDocComment());
        $methodName = $this->convertToLiquid($method->getName());
        $namespace  = $docblock->getTag(Variable\AbstractVariable::DOCBLOCK_TAG_NAMESPACE)
            ? $docblock->getTag(Variable\AbstractVariable::DOCBLOCK_TAG_NAMESPACE)->getDescription()
            : $variable->getNamespace();

        if ((method_exists($variable, $method->getName())
                && $method->getDeclaringClass()->getName() === get_class($variable)
            ) || in_array($method->getName(), $variable->getWhitelist())
        ) {
            $value = '';
            if (
                $docblock->getTag('return')
                && (
                    $docblock->getTag('return')->getType() == 'array'
                    || strpos($docblock->getTag('return')->getType(), '[]') !== false
                )
            ) {
                $value .= "{% for item in {$namespace}.{$methodName} %}";
                $value .= "\\n\\n";
                $value .= "{% endfor %}";
            } else {
                $filter = $docblock->getTag(Variable\AbstractVariable::DOCBLOCK_TAG_FILTER)
                    ? $docblock->getTag(Variable\AbstractVariable::DOCBLOCK_TAG_FILTER)->getDescription()
                    : null;

                $value = "{{ {$namespace}.{$methodName} ";

                if ($filter !== null) {
                    $value .= $filter; // add filter to variable
                }

                $value .= "}}"; // close variable
            }

            $definition = [
                'value' => $value,
                'label' => $docblock->hasTag(Variable\AbstractVariable::DOCBLOCK_TAG_DESCRIPTION)
                    ? $docblock->getTag(Variable\AbstractVariable::DOCBLOCK_TAG_DESCRIPTION)->getDescription()
                    : explode("\n", (string)$docblock->getShortDescription())[0]
            ];
        }

        return $definition;
    }
}
