<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.0
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Ui\Trigger\Form;

use Magento\Framework\App\RequestInterface;
use Magento\Framework\Registry;
use Magento\Ui\DataProvider\AbstractDataProvider;
use Mirasvit\Email\Api\Data\ChainInterface;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Repository\Trigger\ChainRepository;
use Mirasvit\Email\Repository\TriggerRepository;
use Magento\Framework\View\Element\UiComponentFactory;

class DataProvider extends AbstractDataProvider
{
    const TRIGGER = 'trigger';
    const CHAIN   = 'chain';

    private $triggerRepository;

    private $uiComponentFactory;

    /**
     * @var array
     */
    protected $loadedData;

    private $chainRepository;

    private $request;

    private $registry;

    public function __construct(
        Registry $registry,
        RequestInterface $request,
        ChainRepository $chainRepository,
        TriggerRepository $triggerRepository,
        UiComponentFactory $uiComponentFactory,
        string $name,
        string $primaryFieldName,
        string $requestFieldName,
        array $meta = [],
        array $data = []
    ) {
        $this->registry           = $registry;
        $this->request            = $request;
        $this->chainRepository    = $chainRepository;
        $this->triggerRepository  = $triggerRepository;
        $this->collection         = $triggerRepository->getCollection();
        $this->uiComponentFactory = $uiComponentFactory;

        parent::__construct($name, $primaryFieldName, $requestFieldName, $meta, $data);
    }

    /**
     * {@inheritdoc}
     */
    public function getMeta()
    {
        $triggerId = (int)$this->request->getParam($this->getRequestFieldName());
        $trigger = $this->triggerRepository->get($triggerId);

        // Register flags to display rules
        $this->registry->register('event_formName', 'email_trigger_audience', true);
        if ($trigger) {
            $this->registry->register('event_eventIdentifier', $trigger->getEvent(), true);
            $this->registry->register('event_ruleConditions', $trigger->getRule(), true);
        }

        return parent::getMeta();
    }

    /**
     * {@inheritdoc}
     */
    public function getData()
    {
        if (isset($this->loadedData)) {
            return $this->loadedData;
        }

        /** @var TriggerInterface $item */
        foreach ($this->collection as $item) {
            $item = $this->triggerRepository->get((int)$item->getId());
            $data = $item->getData();
            $data[self::CHAIN] = [];
            $data['id_field_name'] = $this->getRequestFieldName();
            $data[TriggerInterface::CANCELLATION_EVENT] = $item->getCancellationEvent(); // convert to array
            unset($data[TriggerInterface::RULE]); // remove dynamic fields
            /** @var ChainInterface $chain */
            foreach ($item->getChainCollection() as $chain) {
                $chain = $this->chainRepository->get($chain->getId());
                $data[self::CHAIN][$chain->getId()] = $chain->getData();
                $data[self::CHAIN][$chain->getId()][ChainInterface::EXCLUDE_DAYS] = $chain->getExcludeDays();
            }

            // set chains for modifier
            $data[ChainInterface::ID] = ['in' => array_keys($data[self::CHAIN])];

            $this->loadedData[$item->getId()] = $data;
            if ($this->request->getParam($this->getRequestFieldName()) === $item->getId()
                && isset($data['report'])
            ) {
                $this->loadedData['report'] = $data['report'];
            }
        }

        return $this->loadedData;
    }
}
