<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.0
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\Email\Setup\Patch\Data;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Framework\Setup\Patch\DataPatchInterface;
use Magento\Framework\Setup\Patch\PatchVersionInterface;
use Mirasvit\Email\Repository\Trigger\ChainRepository;
use Mirasvit\Email\Repository\TriggerRepository;
use Mirasvit\Email\Helper\Data;
use Mirasvit\Email\Helper\Serializer;
use Mirasvit\Email\Model\ResourceModel\Trigger\Chain\Collection as ChainCollection;
use Mirasvit\Email\Model\Trigger\Chain;


class DataPatch106 implements DataPatchInterface, PatchVersionInterface
{
    private $setup;
    private $triggerRepository;
    private $helper;
    private $serializer;

    public function __construct(
        ModuleDataSetupInterface $setup,
        Data $helper,
        TriggerRepository $triggerRepository,
        Serializer               $serializer
    ) {
        $this->setup = $setup;
        $this->triggerRepository = $triggerRepository;
        $this->helper            = $helper;
        $this->serializer        = $serializer;
    }

    /**
     * @inheritdoc
     */
    public function apply()
    {
        $this->setup->getConnection()->startSetup();
        $setup = $this->setup;

        $connection = $setup->getConnection();

        foreach ($this->triggerRepository->getCollection() as $trigger) {
            if ($ruleId = $trigger->getData('run_rule_id')) {
                $select = $connection->select()->from($setup->getTable('mst_email_rule'), ['conditions_serialized'])
                    ->where('rule_id = ?', $ruleId);
                $condition = $connection->fetchOne($select);
                if ($condition) {
                    $condition = $this->serializer->unserialize($condition);

                    $rule['conditions'] = $this->helper->convertConditionsToPost($condition);
                    $conditions = [
                        'Mirasvit\Email\Model\Rule\Condition\Combine'
                        => 'Mirasvit\Event\Model\Rule\Condition\Combine',
                        'Mirasvit\Email\Model\Rule\Condition\Cart'
                        => 'Mirasvit\Event\EventData\Condition\QuoteCondition',
                        'Mirasvit\Email\Model\Rule\Condition\Customer'
                        => 'Mirasvit\Event\EventData\Condition\CustomerCondition',
                        'Mirasvit\Email\Model\Rule\Condition\Order'
                        => 'Mirasvit\Event\EventData\Condition\OrderCondition',
                        'Mirasvit\Email\Model\Rule\Condition\Shipping'
                        => 'Mirasvit\Event\EventData\Condition\AddressShippingCondition',
                        'Mirasvit\Email\Model\Rule\Condition\Wishlist'
                        => 'Mirasvit\Event\EventData\Condition\WishlistCondition',
                        'Mirasvit\Email\Model\Rule\Condition\Product\Subselect'
                        => 'Mirasvit\Event\EventData\Condition\Product\Subselect',
                        'Mirasvit\Email\Model\Rule\Condition\Product\Different'
                        => 'Mirasvit\Event\EventData\Condition\Product\Different',
                    ];

                    // replace condition types
                    foreach ($rule['conditions'] as $key => $condition) {
                        if (isset($condition['type']) && isset($conditions[$condition['type']])) {
                            $rule['conditions'][$key]['type'] = $conditions[$condition['type']];
                        }
                    }

                    $trigger->setRule($rule);

                    $this->triggerRepository->save($trigger);
                }
            }
        }
        $this->setup->getConnection()->endSetup();
    }

    /**
     * Update event names to match event names in module-event.
     */
    private function upgradeEvents()
    {
        $events = [
            'customer_registration'              => 'customer_create',
            'customer_subscription|subscribed'   => 'subscription|subscribed',
            'customer_subscription|unsubscribed' => 'subscription|unsubscribed',
            'cart_abandoned'                     => 'quote_abandoned',
            'review_added'                       => 'review_new',
            'wishlist_itemAdded'                 => 'wishlist_item_new',
        ];

        foreach ($this->triggerRepository->getCollection() as $trigger) {
            $cancellationEvents = [];

            if (isset($events[$trigger->getEvent()])) {
                $trigger->setEvent($events[$trigger->getEvent()]);
            }

            foreach ($trigger->getCancellationEvent() as $cancellationEvent) {
                if (isset($events[$cancellationEvent])) {
                    $cancellationEvent = $events[$cancellationEvent];
                }
                $cancellationEvents[] = $cancellationEvent;
            }

            $trigger->setCancellationEvent($cancellationEvents);

            $this->triggerRepository->save($trigger);
        }
    }

    /**
     * {@inheritdoc}
     */
    public static function getVersion(): string
    {
        return '1.0.6';
    }


    /**
     * {@inheritdoc}
     */
    public static function getDependencies() {
        return [DataPatch104::class];
    }

    /**
     * {@inheritdoc}
     */
    public function getAliases() {
        return [];
    }
}
