<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.0
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Service;

use Magento\Framework\Stdlib\DateTime;
use Magento\Store\Model\StoreManagerInterface;
use Mirasvit\Email\Api\Data\ChainInterface;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Repository\TriggerRepository;
use Mirasvit\Email\Service\EventManagement;
use Mirasvit\Email\Model\Trigger\Handler as TriggerHandler;
use Mirasvit\Event\Api\Repository\EventRepositoryInterface;
use Mirasvit\Event\Api\Service\EventServiceInterface;

class TestSender
{
    private $triggerRepository;

    private $storeManager;

    private $eventService;

    private $eventRepository;

    private $triggerHandler;

    private $eventManagement;

    public function __construct(
        EventManagement $eventManagement,
        TriggerHandler $triggerHandler,
        EventRepositoryInterface $eventRepository,
        EventServiceInterface $eventService,
        StoreManagerInterface $storeManager,
        TriggerRepository $triggerRepository
    ) {
        $this->triggerRepository = $triggerRepository;
        $this->storeManager      = $storeManager;
        $this->eventService      = $eventService;
        $this->eventRepository   = $eventRepository;
        $this->triggerHandler    = $triggerHandler;
        $this->eventManagement   = $eventManagement;
    }

    public function sendChain(ChainInterface $chain, string $to): bool
    {
        $trigger = $this->triggerRepository->get($chain->getTriggerId());

        foreach ($this->getStoreIds($trigger) as $storeId) {
            $event = $this->createEvent($to, (int)$storeId, $trigger->getEvent());

            ini_set('display_errors', '1');

            $queue = $this->triggerHandler->enqueue($trigger, $chain, $event);
            if ($queue) {
                $queue->send();
                $this->eventManagement->addProcessedTriggerId((int)$event->getId(), $trigger->getId());
            }
        }

        return true;
    }

    private function getStoreIds(TriggerInterface $trigger): array
    {
        $storeIds = $trigger->getStoreIds();

        if (empty($storeIds)) {
            /** @var \Magento\Store\Model\Store $store */
            foreach ($this->storeManager->getStores() as $storeId => $store) {
                if ($store->isActive()) {
                    $storeIds[] = (int)$storeId;
                }
            }
        }

        return $storeIds;
    }

    /**
     * Create test event based on passed parameters.
     */
    private function createEvent(string $email, int $storeId, string $event): \Mirasvit\Event\Api\Data\EventInterface
    {
        $params = $this->eventService->getRandomParams($storeId);
        $params['force'] = true;
        $params['is_test'] = true;
        $params['customer_email'] = $email;

        $event = $this->eventRepository->create()
            ->setStoreId($storeId)
            ->setIdentifier($event)
            ->setParams($params->getData())
            ->setKey('test_' . time());

        return $this->eventRepository->save($event);
    }
}
