<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.0
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Model\Trigger;

use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Registry;
use Mirasvit\Email\Api\Data\ChainInterface;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Helper\Data;
use Mirasvit\Email\Model\Config\Source\CrossSell;
use Mirasvit\EmailDesigner\Api\Data\TemplateInterface;
use Mirasvit\EmailDesigner\Repository\TemplateRepository;

/**
 * @method bool   getCouponEnabled()
 * @method int    getCouponSalesRuleId()
 * @method int    getCouponExpiresDays
 */
class Chain extends AbstractModel implements ChainInterface
{
    protected $templateRepository;

    private $helper;

    private $crossSellSource;

    public function __construct(
        TemplateRepository $templateRepository,
        CrossSell $crossSellSource,
        Data $helper,
        Context $context,
        Registry $registry
    ) {
        $this->helper             = $helper;
        $this->templateRepository = $templateRepository;

        parent::__construct($context, $registry);

        $this->crossSellSource = $crossSellSource;
    }

    /**
     * {@inheritdoc}
     */
    protected function _construct()
    {
        $this->_init(\Mirasvit\Email\Model\ResourceModel\Trigger\Chain::class);
    }

    public function getTemplate(): TemplateInterface
    {
        return $this->templateRepository->get($this->getTemplateId());
    }

    public function getDay(bool $inSeconds = false): int
    {
        $day = (int)$this->getData(ChainInterface::DAY);
        if ($inSeconds) {
            $day *= 60 * 60 * 24;
        }

        return $day;
    }

    public function getHour(bool $inSeconds = false): int
    {
        $hours = (int)$this->getData(ChainInterface::HOUR);
        if ($inSeconds) {
            $hours *= 60 * 60;
        }

        return $hours;
    }

    public function getMinute(bool $inSeconds = false): int
    {
        $minutes = (int)$this->getData(ChainInterface::MINUTE);
        if ($inSeconds) {
            $minutes *= 60;
        }

        return $minutes;
    }

    /**
     * @todo RF
     *
     * {@inheritdoc}
     */
    public function getScheduledAt(int $time): int
    {
        $excludeDays = $this->getExcludeDays();
        $frequency = $this->getDay(true);
        $hours = $this->getHour(true);
        $minutes = $this->getMinute(true);

        $scheduledAt = $time + $frequency + $hours + $minutes;

        $scheduledAt = $scheduledAt + $this->addExcludedDays($scheduledAt, $excludeDays) * 86400;

        return $scheduledAt;
    }

    protected function addExcludedDays(int $time, array $excludeDaysOfWeek = null): int
    {
        $result = 0;
        if (is_array($excludeDaysOfWeek) && (count($excludeDaysOfWeek) > 0)) {
            while (in_array(date('w', $time + $result * 86400), $excludeDaysOfWeek)) {
                ++$result;

                if ($result > 7) {
                    break;
                }
            }
        }

        return $result;
    }

    public function getId(): int
    {
        return (int)$this->getData(self::ID);
    }

    public function setTemplateId(string $id): ChainInterface
    {
        if (is_numeric($id)) {
            // add EmailDesigner prefix by default
            $id = \Mirasvit\EmailDesigner\Model\Template::TYPE . ':' . $id;
        }

        $this->setData(self::TEMPLATE_ID, $id);

        return $this;
    }

    public function getTemplateId(): string
    {
        return (string)$this->getData(self::TEMPLATE_ID);
    }

    public function getTriggerId(): int
    {
        return (int)$this->getData(TriggerInterface::ID);
    }

    public function setTriggerId(int $id): ChainInterface
    {
        $this->setData(TriggerInterface::ID, $id);

        return $this;
    }

    public function setDay(int $day): ChainInterface
    {
        $this->setData(self::DAY, $day);

        return $this;
    }

    public function setHour(int $hour): ChainInterface
    {
        $this->setData(self::HOUR, $hour);

        return $this;
    }

    public function setMinute(int $minute): ChainInterface
    {
        $this->setData(self::MINUTE, $minute);

        return $this;
    }

    public function getSendFrom(): int
    {
        return (int)$this->getData(self::SEND_FROM);
    }

    public function getSendTo(): int
    {
        return (int)$this->getData(self::SEND_TO);
    }

    public function setSendFrom(int $sendFrom): ChainInterface
    {
        $this->setData(self::SEND_FROM, $sendFrom);

        return $this;
    }

    public function setSendTo(int $sendTo): ChainInterface
    {
        $this->setData(self::SEND_TO, $sendTo);

        return $this;
    }

    public function getExcludeDays(): ?array
    {
        $excludeDays = $this->getData(self::EXCLUDE_DAYS);
        if (is_string($excludeDays)) {
            $excludeDays = explode(',', $excludeDays);
        }

        return $excludeDays ?: null;
    }

    public function setExcludeDays(array $excludeDays = null): ChainInterface
    {
        if (!$excludeDays) {
            return $this;
        }

        if (is_array($excludeDays)) {
            $excludeDays = implode(',', $excludeDays);
        }

        $this->setData(self::EXCLUDE_DAYS, $excludeDays);

        return $this;
    }

    public function getCrossSellsEnabled(): bool
    {
        return (bool)$this->getData(self::CROSS_SELLS_ENABLED);
    }

    public function getCrossSellsTypeId(): string
    {
        return (string)$this->getData(self::CROSS_SELLS_TYPE_ID);
    }

    public function getCrossSellMethodName(): ?string
    {
        $methodName = null;

        switch ($this->getCrossSellsTypeId()) {
            case CrossSell::MAGE_CROSS:
                $methodName = 'getCrossSellProductIds';
                break;
            case CrossSell::MAGE_RELATED:
                $methodName = 'getRelatedProductIds';
                break;
            case CrossSell::MAGE_UPSELLS:
                $methodName = 'getUpSellProductIds';
                break;
        }

        return $methodName;
    }

    /**
     * {@inheritDoc}
     *
     * @SuppressWarnings(PHPMD)
     */
    public function toString($format = '')
    {
        $delay = '';

        if ($this->getDay()) {
            $delay .= " <b>{$this->getDay()}</b>" . ($this->getDay() > 1 ? ' days' : 'day');
        }

        if ($this->getHour()) {
            $delay .= " <b>{$this->getHour()}</b>" . ($this->getHour() > 1 ? ' hours' : ' hour');
        }
        if ($this->getMinute()) {
            $delay .= " <b>{$this->getMinute()}</b>" . ($this->getMinute() > 1 ? ' mins' : ' min');
        }

        if (!$this->getDay() && !$this->getHour() && !$this->getMinute()) {
            $delay = 'immediately';
        } else {
            $delay = 'delay for '.$delay;
        }

        $coupon = '';
        if ($this->getCouponEnabled()) {
            $coupon = 'with coupon';
        }

        $crossSellSource = '';
        if ($this->getCrossSellsEnabled() && $this->getCrossSellsTypeId()) {
            $crossSellSource = ' including ' . $this->crossSellSource->toArray()[$this->getCrossSellsTypeId()];
        }

        if ($this->getTemplate()) {
            return (string)__(
                '%1 %2 %3',
                $delay,
                $coupon,
                $crossSellSource
            );
        } else {
            return (string)__(
                'No Template Selected - <small>the previously used email template has likely been removed</small>.'
            );
        }
    }
}
