<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.4.9
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Registry;
use Magento\Framework\Stdlib\DateTime;
use Magento\Store\Model\StoreManagerInterface;
use Mirasvit\Email\Api\Data\CampaignInterface;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Repository\Trigger\ChainRepository;
use Mirasvit\Email\Model\Trigger\Handler;
use Mirasvit\Email\Model\ResourceModel\Trigger\Chain\Collection;
use Mirasvit\Event\Api\Repository\EventRepositoryInterface;
use Mirasvit\Event\Api\Service\ValidatorServiceInterface;
use Mirasvit\Event\Service\EventService;
use Mirasvit\Core\Service\SerializeService;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Trigger extends AbstractModel implements TriggerInterface
{
    /**
     * @var string
     */
    protected $_eventPrefix = 'email_trigger';

    protected $chainCollection;

    protected $chainRepository;

    protected $storeManager;

    protected $scopeConfig;

    protected $handler;

    protected $context;

    protected $registry;

    private $eventService;

    private $eventRepository;

    private $validatorService;

    public function __construct(
        ValidatorServiceInterface $validatorService,
        EventRepositoryInterface $eventRepository,
        EventService $eventService,
        Handler $handler,
        ChainRepository $chainRepository,
        StoreManagerInterface $storeManager,
        ScopeConfigInterface $scopeConfig,
        Context $context,
        Registry $registry
    ) {
        $this->validatorService = $validatorService;
        $this->handler          = $handler;
        $this->eventRepository  = $eventRepository;
        $this->chainRepository  = $chainRepository;
        $this->eventService     = $eventService;
        $this->storeManager     = $storeManager;
        $this->scopeConfig      = $scopeConfig;
        $this->context          = $context;
        $this->registry         = $registry;

        parent::__construct($context, $registry);
    }

    /**
     * {@inheritdoc}
     */
    protected function _construct()
    {
        $this->_init(\Mirasvit\Email\Model\ResourceModel\Trigger::class);
    }

    public function getId()
    {
        return (int)$this->getData(self::ID);
    }

    public function getChainCollection(): Collection
    {
        if ($this->chainCollection == null) {
            $this->chainCollection = $this->chainRepository->getCollection()
                ->addFieldToFilter(self::ID, $this->getId());
        }

        return $this->chainCollection;
    }

    public function getTriggeringEvents(): array
    {
        return [$this->getData(self::EVENT)];
    }

    public function getCancellationEvents(): array
    {
        return $this->getCancellationEvent();
    }

    public function getEvents(): array
    {
        return array_values(array_unique(array_merge($this->getTriggeringEvents(), $this->getCancellationEvents())));
    }

    public function getRuleSerialized(): string
    {
        return (string)$this->getData(self::RULE_SERIALIZED);
    }

    public function setRuleSerialized(string $value): TriggerInterface
    {
        return $this->setData(self::RULE_SERIALIZED, $value);
    }

    /**
     * {@inheritDoc}
     */
    public function getRule(): array
    {
        if (!$this->hasData(self::RULE) && $this->getRuleSerialized()) {
            $data = SerializeService::decode($this->getRuleSerialized());
            $this->setData(self::RULE, $data);
        } elseif (!$this->hasData(self::RULE) && !$this->getRuleSerialized()) {
            $this->setData(self::RULE, []);
        }

        return $this->getData(self::RULE);
    }

    /**
     * {@inheritDoc}
     */
    public function setRule($rule): TriggerInterface
    {
        $this->setData(self::RULE, $rule);

        return $this;
    }

    public function getSenderEmail(int $storeId = 0): string
    {
        if ($this->getData(self::SENDER_EMAIL)) {
            return (string)$this->getData(self::SENDER_EMAIL);
        }

        return (string)$this->scopeConfig->getValue(
            'trans_email/ident_general/email',
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    public function getSenderName(int $storeId = 0): string
    {
        if ($this->getData(self::SENDER_NAME)) {
            return (string)$this->getData(self::SENDER_NAME);
        }

        return (string)$this->scopeConfig->getValue(
            'trans_email/ident_general/name',
            ScopeInterface::SCOPE_STORE,
            $storeId
        );
    }

    /**
     * Send test email
     */
    public function sendTest(string $to = null): bool
    {
        $storeIds = $this->getStoreIds();
        if ($storeIds[0] == 0) {
            unset($storeIds[0]);
            foreach ($this->storeManager->getStores() as $storeId => $store) {
                if ($store->getIsActive()) {
                    $storeIds[] = $storeId;
                }
            }
        }

        foreach ($storeIds as $storeId) {
            $params = $this->eventService->getRandomParams($storeId);

            if ($to) {
                $params->setData('customer_email', $to);
            }

            $params['force'] = true;
            $params['is_test'] = true;

            $event = $this->eventRepository->create()
                ->setStoreId($storeId)
                ->setIdentifier($this->getEvent())
                ->setParams($params->getData())
                ->setKey('test_' . time())
                ->setUpdatedAt((new \DateTime())->format(DateTime::DATETIME_PHP_FORMAT))
                ->setCreatedAt((new \DateTime())->format(DateTime::DATETIME_PHP_FORMAT));

            $this->eventRepository->save($event);

            ini_set('display_errors', 1);

            $this->handler->triggerEvent($this, $event);
        }

        return true;
    }

    /**
     * Validate event args by trigger rules
     */
    public function validateRules(array $args, bool $force = false): bool
    {
        if (isset($args['force']) && !$force) {
            return true;
        }

        $eventInstance = $this->eventRepository->getInstance($this->getEvent());

        $result = $this->validatorService->validate($this->getRule(), $eventInstance->expand($args));

        return $result;
    }

    public function getTitle(): string
    {
        return (string)$this->getData(self::TITLE);
    }

    public function setTitle(string $title): TriggerInterface
    {
        $this->setData(self::TITLE, $title);

        return $this;
    }

    public function getDescription(): string
    {
        return (string)$this->getData(self::DESCRIPTION);
    }

    public function setDescription(string $description): TriggerInterface
    {
        $this->setData(self::DESCRIPTION, $description);

        return $this;
    }

    public function getSchedule(): string
    {
        return (string)$this->getData(self::SCHEDULE);
    }

    public function setSchedule(string $schedule): TriggerInterface
    {
        $this->setData(self::SCHEDULE, $schedule);

        return $this;
    }

    public function getTriggerType(): string
    {
        return (string)$this->getData(self::TRIGGER_TYPE);
    }

    public function setTriggerType(string $triggerType): TriggerInterface
    {
        $this->setData(self::TRIGGER_TYPE, $triggerType);

        return $this;
    }

    public function getEvent(): string
    {
        return (string)$this->getData(self::EVENT);
    }

    public function setEvent(string $event): TriggerInterface
    {
        $this->setData(self::EVENT, $event);

        return $this;
    }

    public function getCancellationEvent(): array
    {
        $cancellationEvent = $this->getData(self::CANCELLATION_EVENT);
        if (!is_array($cancellationEvent)) {
            $cancellationEvent = array_filter(explode(',', (string)$cancellationEvent));
        }

        return $cancellationEvent;
    }

    public function setCancellationEvent(array $cancellationEvent): TriggerInterface
    {
        if (is_array($cancellationEvent)) {
            $cancellationEvent = implode(',', $cancellationEvent);
        }

        $this->setData(self::CANCELLATION_EVENT, $cancellationEvent);

        return $this;
    }

    public function getGaSource(): string
    {
        return (string)$this->getData(self::GA_SOURCE);
    }

    public function setGaSource(string $source): TriggerInterface
    {
        $this->setData(self::GA_SOURCE, $source);

        return $this;
    }

    public function getGaMedium(): string
    {
        return (string)$this->getData(self::GA_MEDIUM);
    }

    public function setGaMedium(string $medium): TriggerInterface
    {
        $this->setData(self::GA_MEDIUM, $medium);

        return $this;
    }

    public function getGaName(): string
    {
        return (string)$this->getData(self::GA_NAME);
    }

    public function setGaName(string $name): TriggerInterface
    {
        $this->setData(self::GA_NAME, $name);

        return $this;
    }

    public function getGaTerm(): string
    {
        return (string)$this->getData(self::GA_TERM);
    }

    public function setGaTerm(string $term): TriggerInterface
    {
        $this->setData(self::GA_TERM, $term);

        return $this;
    }

    public function getGaContent(): string
    {
        return (string)$this->getData(self::GA_CONTENT);
    }

    public function setGaContent(string $content): TriggerInterface
    {
        $this->setData(self::GA_CONTENT, $content);

        return $this;
    }

    public function getStoreIds(): array
    {
        $storeIds = $this->getData(self::STORE_IDS);
        if (is_string($storeIds)) {
            $storeIds = explode(',', $storeIds);
        } elseif ($storeIds == null) {
            return [];
        }

        return array_filter($storeIds);
    }

    public function setStoreIds(array $storeIds): TriggerInterface
    {
        if (is_array($storeIds)) {
            $storeIds = implode(',', $storeIds);
        }

        $this->setData(self::STORE_IDS, $storeIds);

        return $this;
    }

    public function setCampaignId(string $campaignId): TriggerInterface
    {
        $this->setData(CampaignInterface::ID, $campaignId);

        return $this;
    }

    public function getCampaignId(): string
    {
        return (string)$this->getData(CampaignInterface::ID);
    }

    public function getIsActive(): bool
    {
        return (bool)$this->getData(self::IS_ACTIVE);
    }

    public function setIsActive(bool $isActive): TriggerInterface
    {
        $this->setData(self::IS_ACTIVE, $isActive);

        return $this;
    }

    public function setSenderEmail(string $senderEmail): TriggerInterface
    {
        $this->setData(self::SENDER_EMAIL, $senderEmail);

        return $this;
    }

    public function setSenderName(string $senderName): TriggerInterface
    {
        $this->setData(self::SENDER_NAME, $senderName);

        return $this;
    }

    public function getActiveTo(): string
    {
        return (string)$this->getData(self::ACTIVE_TO);
    }

    public function setActiveTo(string $activeTo): TriggerInterface
    {
        $this->setData(self::ACTIVE_TO, $activeTo);

        return $this;
    }

    public function getActiveFrom(): string
    {
        return (string)$this->getData(self::ACTIVE_FROM);
    }

    public function setActiveFrom(string $activeFrom): TriggerInterface
    {
        $this->setData(self::ACTIVE_FROM, $activeFrom);

        return $this;
    }

    public function getCreatedAt(): string
    {
        return (string)$this->getData(self::CREATED_AT);
    }

    public function setCreatedAt(string $createdAt): TriggerInterface
    {
        $this->setData(self::CREATED_AT, $createdAt);

        return $this;
    }

    public function getUpdatedAt(): string
    {
        return (string)$this->getData(self::UPDATED_AT);
    }

    public function setUpdatedAt(string $updatedAt): TriggerInterface
    {
        $this->setData(self::UPDATED_AT, $updatedAt);

        return $this;
    }

    public function getIsAdmin(): int
    {
        return (int)$this->getData(self::IS_ADMIN);
    }

    public function setIsAdmin(int $value): TriggerInterface
    {
        return $this->setData(self::IS_ADMIN, $value);
    }

    public function getAdminEmail(): string
    {
        return (string)$this->getData(self::ADMIN_EMAIL);
    }

    public function setAdminEmail(string $email): TriggerInterface
    {
        if (empty($email)) {
            $this->setIsAdmin(self::IS_ADMIN_DISABLED);
        } else {
            $this->setIsAdmin(self::IS_ADMIN_ACTIVE);
        }

        return $this->setData(self::ADMIN_EMAIL, $email);
    }
}
