<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.4.9
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\Email\Model;

use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Registry;
use Magento\Framework\Stdlib\DateTime;
use Mirasvit\Email\Api\Data\QueueInterface;
use Mirasvit\Email\Api\Data\TriggerInterface;
use Mirasvit\Email\Service\ErrorHandler;
use Mirasvit\Email\Model\Queue\Sender;
use Mirasvit\Email\Model\Trigger\ChainFactory;
use Mirasvit\EmailDesigner\Api\Data\TemplateInterface;
use Mirasvit\EmailDesigner\Service\TemplateProcessor;
use Mirasvit\Email\Helper\Serializer;

/**
 * @method bool hasStatus()
 * @method $this setStatusMessage($message)
 * @method bool hasCreatedAt()
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class Queue extends AbstractModel implements QueueInterface
{
    /**
     * @var array
     */
    protected $args;

    protected $trigger;

    protected $emailChain;

    protected $triggerFactory;

    protected $chainFactory;

    protected $sender;

    private $errorHandler;

    private $templateProcessor;

    private $serializer;

    public function __construct(
        TemplateProcessor $templateProcessor,
        ErrorHandler $errorHandler,
        Sender $sender,
        TriggerFactory $triggerFactory,
        ChainFactory $chainFactory,
        Context $context,
        Registry $registry,
        Serializer $serializer
    ) {
        $this->templateProcessor = $templateProcessor;
        $this->errorHandler      = $errorHandler;
        $this->triggerFactory    = $triggerFactory;
        $this->chainFactory      = $chainFactory;
        $this->sender            = $sender;
        $this->serializer        = $serializer;

        parent::__construct($context, $registry);
    }

    /**
     * {@inheritdoc}
     */
    protected function _construct()
    {
        $this->_init(\Mirasvit\Email\Model\ResourceModel\Queue::class);
    }

    public function getTrigger(): TriggerInterface
    {
        if ($this->trigger == null) {
            $this->trigger = $this->triggerFactory->create()
                ->load($this->getTriggerId());
        }

        return $this->trigger;
    }

    public function getChain(): AbstractModel
    {
        if ($this->emailChain === null) {
            $this->emailChain = $this->chainFactory->create()
                ->load($this->getChainId());
        }

        return $this->emailChain;
    }

    public function getTemplate(): TemplateInterface
    {
        return $this->getChain()->getTemplate();
    }

    /**
     * Get queue args.
     *
     * @param string $key
     *
     * @return array|string
     */
    public function getArgs(string $key = null)
    {
        if ($this->args == null) {
            $this->args = $this->serializer->unserialize($this->getData('args_serialized'));
            $this->args['trigger'] = $this->getTrigger();
            $this->args['chain'] = $this->getChain();
            $this->args['queue'] = $this;
        }

        if ($key) {
            if (isset($this->args[$key])) {
                return $this->args[$key];
            } else {
                return null;
            }
        }

        return $this->args;
    }

    public function getArg(string $key): ?string
    {
        return $this->getArgs($key) ? (string)$this->getArgs($key) : null;
    }

    public function getMailSubject(): string
    {
        if (!$this->getSubject()) {
            $this->errorHandler->registerErrorHandler();
            $this->setSubject($this->templateProcessor->processSubject($this->getTemplate(), $this->getArgs()));

            if (isset($this->args['is_test'])) {
                $subject = $this->getSubject()
                    . ' ' . (string)__('[Test Mail. Store #%1 %2]', $this->args['store_id'], date('d/m/Y H:i:s'));

                $this->errorHandler->restoreErrorHandler();

                return $subject;
            }

            $this->errorHandler->restoreErrorHandler();
        }

        return $this->getSubject();
    }

    public function getMailContent(): string
    {
        if (!$this->getContent()) {
            $this->errorHandler->registerErrorHandler();

            $this->setContent($this->templateProcessor->processTemplate($this->getTemplate(), $this->getArgs()));

            $this->errorHandler->restoreErrorHandler();
        }

        return $this->getContent();
    }

    public function send(bool $force = false): QueueInterface
    {
        $this->sender->send($this, $force);

        return $this;
    }

    public function pending(string $message = ''): QueueInterface
    {
        $this->setStatus(self::STATUS_PENDING)
            ->setStatusMessage($message)
            ->save();

        return $this;
    }

    public function delivery(string $message = ''): QueueInterface
    {
        $this->setSentAt((new \DateTime())->format(DateTime::DATETIME_PHP_FORMAT))
            ->setStatus(self::STATUS_SENT)
            ->setStatusMessage($message)
            ->save();

        $this->_eventManager->dispatch('email_queue_delivery', ['queue' => $this]);

        return $this;
    }

    public function miss(string $message = ''): QueueInterface
    {
        $this->setStatus(self::STATUS_MISSED)
            ->setStatusMessage($message)
            ->save();

        return $this;
    }

    public function cancel(string $message = ''): QueueInterface
    {
        $this->setStatus(self::STATUS_CANCELED)
            ->setStatusMessage($message)
            ->save();

        return $this;
    }

    public function error(string $message = ''): QueueInterface
    {
        $this->setStatus(self::STATUS_ERROR)
            ->setStatusMessage($message)
            ->save();

        return $this;
    }

    public function unsubscribe(string $message = ''): QueueInterface
    {
        $this->setStatus(self::STATUS_UNSUBSCRIBED)
            ->setStatusMessage($message)
            ->save();

        return $this;
    }

    public function reset(string $message = ''): QueueInterface
    {
        $this->setStatus(self::STATUS_PENDING)
            ->setStatusMessage($message)
            ->setSentAt(null)
            ->setContent(null)
            ->save();

        return $this;
    }

    public function getTriggerId(): int
    {
        return (int)$this->getData(self::TRIGGER_ID);
    }

    public function setTriggerId(int $triggerId): QueueInterface
    {
        $this->setData(self::TRIGGER_ID, $triggerId);

        return $this;
    }

    public function getStatus(): string
    {
        return (string)$this->getData(self::STATUS);
    }

    public function setStatus(string $status): QueueInterface
    {
        $this->setData(self::STATUS, $status);

        return $this;
    }

    public function getChainId(): int
    {
        return (int)$this->getData(self::CHAIN_ID);
    }

    public function setChainId(int $chainId): QueueInterface
    {
        $this->setData(self::CHAIN_ID, $chainId);

        return $this;
    }

    public function getUniqKey(): string
    {
        return (string)$this->getData(self::UNIQUE_KEY);
    }

    public function setUniqKey(string $uniqKey): QueueInterface
    {
        $this->setData(self::UNIQUE_KEY, $uniqKey);

        return $this;
    }

    public function getUniqHash(): string
    {
        return (string)$this->getData(self::UNIQUE_HASH);
    }

    public function setUniqHash(string $uniqHash): QueueInterface
    {
        $this->setData(self::UNIQUE_HASH, $uniqHash);

        return $this;
    }

    public function getScheduledAt(): string
    {
        return (string)$this->getData(self::SCHEDULED_AT);
    }

    public function setScheduledAt(string $scheduledAt): QueueInterface
    {
        $this->setData(self::SCHEDULED_AT, $scheduledAt);

        return $this;
    }

    public function getSentAt(): ?string
    {
        return $this->getData(self::SENT_AT) ? (string)$this->getData(self::SENT_AT) : null;
    }

    public function setSentAt(string $sentAt = null): QueueInterface
    {
        $this->setData(self::SENT_AT, $sentAt);

        return $this;
    }

    public function getAttemtpsNumber(): int
    {
        return (int)$this->getData(self::ATTEMPTS_NUMBER);
    }

    public function setAttemtpsNumber(int $attemtpsNumber): QueueInterface
    {
        $this->setData(self::ATTEMPTS_NUMBER, $attemtpsNumber);

        return $this;
    }

    public function getSenderEmail(): string
    {
        return (string)$this->getData(self::SENDER_EMAIL);
    }

    public function setSenderEmail(string $senderEmail): QueueInterface
    {
        $this->setData(self::SENDER_EMAIL, $senderEmail);

        return $this;
    }

    public function getSenderName(): string
    {
        return (string)$this->getData(self::SENDER_NAME);
    }

    public function setSenderName(string $senderName): QueueInterface
    {
        $this->setData(self::SENDER_NAME, $senderName);

        return $this;
    }

    public function getRecipientEmail(): string
    {
        return (string)$this->getData(self::RECIPIENT_EMAIL);
    }

    public function setRecipientEmail(string $recipientEmail): QueueInterface
    {
        $this->setData(self::RECIPIENT_EMAIL, $recipientEmail);

        return $this;
    }

    public function getRecipientName(): string
    {
        return (string)$this->getData(self::RECIPIENT_NAME);
    }

    public function setRecipientName(string $recipientName): QueueInterface
    {
        $this->setData(self::RECIPIENT_NAME, $recipientName);

        return $this;
    }

    public function getContent(): ?string
    {
        return $this->getData(self::CONTENT) ? (string)$this->getData(self::CONTENT) : null;
    }

    public function setContent(string $content = null): QueueInterface
    {
        $this->setData(self::CONTENT, $content);

        return $this;
    }

    public function getSubject(): string
    {
        return (string)$this->getData(self::SUBJECT);
    }

    public function setSubject(string $subject): QueueInterface
    {
        $this->setData(self::SUBJECT, $subject);

        return $this;
    }

    public function setArgs(array $args): QueueInterface
    {
        $this->setData(self::ARGS, $args);

        return $this;
    }

    public function getArgsSerialized(): string
    {
        return (string)$this->getData(self::ARGS_SERIALIZED);
    }

    public function setArgsSerialized(string $argsSerialized): QueueInterface
    {
        $this->setData(self::ARGS_SERIALIZED, $argsSerialized);

        return $this;
    }

    public function getHistory(): string
    {
        return (string)$this->getData(self::HISTORY);
    }

    public function setHistory(string $history): QueueInterface
    {
        $this->setData(self::HISTORY, $history);

        return $this;
    }

    public function getUpdatedAt(): string
    {
        return (string)$this->getData(self::UPDATED_AT);
    }

    public function setUpdatedAt(string $updatedAt): QueueInterface
    {
        $this->setData(self::UPDATED_AT, $updatedAt);

        return $this;
    }

    public function getCreatedAt(): string
    {
        return (string)$this->getData(self::CREATED_AT);
    }

    public function setCreatedAt(string $createdAt): QueueInterface
    {
        $this->setData(self::CREATED_AT, $createdAt);

        return $this;
    }
}
