<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-email
 * @version   2.5.3
 * @copyright Copyright (C) 2023 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\Email\Ui\Event\Listing;

use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\Search\ReportingInterface;
use Magento\Framework\Api\Search\SearchCriteriaBuilder;
use Magento\Framework\Api\Search\SearchResultInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\View\Element\UiComponent\DataProvider\Document;
use Mirasvit\Core\Service\SerializeService;
use Mirasvit\Email\Repository\EventRepository;
use Mirasvit\Email\Service\EventManagement;
use Mirasvit\Event\Api\Data\EventInterface;

/**
 * @SuppressWarnings(PHPMD)
 */
class DataProvider extends \Magento\Framework\View\Element\UiComponent\DataProvider\DataProvider
{
    private $eventRepository;

    private $eventManagement;

    public function __construct(
        EventRepository       $eventRepository,
        EventManagement       $eventManagement,
        string                $name,
        string                $primaryFieldName,
        string                $requestFieldName,
        ReportingInterface    $reporting,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        RequestInterface      $request,
        FilterBuilder         $filterBuilder,
        array                 $meta = [],
        array                 $data = []
    ) {
        $this->eventRepository = $eventRepository;
        $this->eventManagement = $eventManagement;

        parent::__construct($name, $primaryFieldName, $requestFieldName, $reporting, $searchCriteriaBuilder, $request, $filterBuilder, $meta, $data);
    }

    /**
     * {@inheritdoc}
     */
    protected function searchResultToOutput(SearchResultInterface $searchResult)
    {
        $result = [
            'items'        => [],
            'totalRecords' => $searchResult->getTotalCount(),
        ];

        /** @var Document $item */
        foreach ($searchResult->getItems() as $item) {
            $this->addMessage($item);
            $this->addParams($item);
            $this->addTriggers($item);

            $result['items'][] = $item->getData();
        }

        return $result;
    }

    private function addMessage(Document $item): void
    {
        $params     = SerializeService::decode($item->getData(EventInterface::PARAMS_SERIALIZED));
        $identifier = $item->getData(EventInterface::IDENTIFIER);

        $event = $this->eventRepository->getInstance($identifier);

        $string = $event->toString($params);

        $item->setData('message', '<div style="white-space: normal; max-height: 20rem; overflow: scroll">' . $string . '</div>');
    }

    private function addParams(Document $item): void
    {
        $params = SerializeService::decode($item->getData(EventInterface::PARAMS_SERIALIZED));

        ksort($params);

        $html = '<table class="email__sub-table">';
        $html .= '<tr><td>unique_key</td><td>' . $item->getData('key') . '</td></tr>';
        foreach ($params as $key => $value) {
            if (is_scalar($value)) {
                $value = (string)$value;

                $html .= '<tr>';
                $html .= '<td>' . $key . '</td>';
                $html .= '<td>' . (strlen($value) > 150
                        ? rtrim(substr($value, 0, 150)) . '...'
                        : $value
                    ) . '</td>';
                $html .= '</tr> ';
            }
        }
        $html .= '</table > ';

        $item->setData('params', $html);
    }

    private function addTriggers(Document $item): void
    {
        $triggerIds = $this->eventManagement->getProcessedTriggerIds((int)$item->getData(EventInterface::ID));

        $arr = [];
        foreach ($triggerIds as $info) {
            $arr[$info['trigger_id']] = $info['status'];
        }

        ksort($arr);

        $html = '<table class="email__sub-table">';
        foreach ($arr as $key => $value) {
            $html .= '<tr>';
            $html .= '<td>' . $key . '</td>';
            $html .= '<td>' . $value . '</td>';
            $html .= '</tr>';
        }
        $html .= '</table>';

        $item->setData('triggers', $html);
    }
}
