<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_GiftCard
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\GiftCard\Setup;

use Magento\Catalog\Model\Category;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\Product\AttributeSet\Options;
use Magento\Catalog\Setup\CategorySetupFactory;
use Magento\Cms\Model\BlockFactory;
use Magento\Eav\Model\Entity\Attribute\ScopedAttributeInterface;
use Magento\Eav\Setup\EavSetupFactory;
use Magento\Framework\Setup\InstallDataInterface;
use Magento\Framework\Setup\ModuleContextInterface;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Quote\Setup\QuoteSetupFactory;
use Magento\Sales\Setup\SalesSetupFactory;
use Mageplaza\GiftCard\Model\Product\Type\GiftCard;
use Mageplaza\GiftCard\Model\TemplateFactory;

/**
 * @codeCoverageIgnore
 */
class InstallData implements InstallDataInterface
{
    /**
     * @type \Magento\Eav\Setup\EavSetupFactory
     */
    protected $eavSetupFactory;

    /**
     * @var SalesSetupFactory
     */
    protected $salesSetupFactory;

    /**
     * @var \Magento\Quote\Setup\QuoteSetupFactory
     */
    protected $quoteSetupFactory;

    /**
     * @var \Magento\Catalog\Setup\CategorySetupFactory
     */
    protected $categorySetupFactory;

    /**
     * @var \Mageplaza\GiftCard\Setup\BlockFactory
     */
    protected $blockFactory;

    /**
     * @var \Mageplaza\GiftCard\Setup\TemplateFactory
     */
    protected $templateFactory;

    /**
     * @var \Magento\Catalog\Model\Product\AttributeSet\Options
     */
    protected $_attributeSet;

    /**
     * Install Data constructor.
     *
     * @param \Magento\Eav\Setup\EavSetupFactory $eavSetupFactory
     * @param \Magento\Sales\Setup\SalesSetupFactory $salesSetupFactory
     * @param \Magento\Quote\Setup\QuoteSetupFactory $quoteSetupFactory
     * @param \Magento\Catalog\Setup\CategorySetupFactory $categorySetupFactory
     * @param \Magento\Cms\Model\BlockFactory $blockFactory
     * @param \Mageplaza\GiftCard\Model\TemplateFactory $templateFactory
     * @param \Magento\Catalog\Model\Product\AttributeSet\Options $attributeSet
     */
    public function __construct(
        EavSetupFactory $eavSetupFactory,
        SalesSetupFactory $salesSetupFactory,
        QuoteSetupFactory $quoteSetupFactory,
        CategorySetupFactory $categorySetupFactory,
        BlockFactory $blockFactory,
        TemplateFactory $templateFactory,
        Options $attributeSet
    )
    {
        $this->eavSetupFactory      = $eavSetupFactory;
        $this->salesSetupFactory    = $salesSetupFactory;
        $this->quoteSetupFactory    = $quoteSetupFactory;
        $this->categorySetupFactory = $categorySetupFactory;
        $this->blockFactory         = $blockFactory;
        $this->templateFactory      = $templateFactory;
        $this->_attributeSet        = $attributeSet;
    }

    /**
     * {@inheritdoc}
     * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
     */
    public function install(ModuleDataSetupInterface $setup, ModuleContextInterface $context)
    {
        $installer = $setup;

        /** @var \Magento\Sales\Setup\SalesSetup $salesInstaller */
        $salesInstaller = $this->salesSetupFactory->create(['resourceName' => 'sales_setup', 'setup' => $setup]);

        /** @var \Magento\Quote\Setup\QuoteSetup $quoteInstaller */
        $quoteInstaller = $this->quoteSetupFactory->create(['resourceName' => 'quote_setup', 'setup' => $setup]);

        /** @var \Magento\Catalog\Setup\CategorySetup $catalogSetup */
        $catalogSetup = $this->categorySetupFactory->create(['setup' => $setup]);

        $installer->startSetup();

        /** Add field to process gift card discount amount */
        $salesInstaller->addAttribute('order', 'gift_cards', ['type' => 'text']);
        $salesInstaller->addAttribute('order', 'base_gift_card_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('order', 'gift_card_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('order', 'base_gift_credit_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('order', 'gift_credit_amount', ['type' => 'decimal']);

        $salesInstaller->addAttribute('invoice', 'base_gift_card_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('invoice', 'gift_card_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('invoice', 'base_gift_credit_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('invoice', 'gift_credit_amount', ['type' => 'decimal']);

        $salesInstaller->addAttribute('creditmemo', 'base_gift_card_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('creditmemo', 'gift_card_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('creditmemo', 'base_gift_credit_amount', ['type' => 'decimal']);
        $salesInstaller->addAttribute('creditmemo', 'gift_credit_amount', ['type' => 'decimal']);

        $quoteInstaller->addAttribute('quote', 'gift_cards', ['type' => 'text']);
        $quoteInstaller->addAttribute('quote', 'gc_credit', ['type' => 'text']);

        /** Create product attribute group */
        $entityTypeId = $catalogSetup->getEntityTypeId(Category::ENTITY);

        foreach ($this->_attributeSet->toOptionArray() as $set) {
            $catalogSetup->addAttributeGroup($entityTypeId, $set['value'], 'Gift Card Information', 10);
        }

        /** Add Product Attribute */
        $catalogSetup->addAttribute(Product::ENTITY, 'gift_code_pattern', array_merge($this->getDefaultOptions(), [
            'label'          => 'Gift Code Pattern',
            'type'           => 'varchar',
            'input'          => 'text',
            'backend'        => 'Mageplaza\GiftCard\Model\Attribute\Backend\Pattern',
            'input_renderer' => 'Mageplaza\GiftCard\Block\Adminhtml\Product\Helper\Form\Config\Pattern',
            'required'       => true,
            'sort_order'     => 10
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'gift_card_type', array_merge($this->getDefaultOptions(), [
            'label'      => 'Gift Card Type',
            'type'       => 'int',
            'input'      => 'select',
            'source'     => 'Mageplaza\GiftCard\Model\Product\DeliveryMethods',
            'global'     => ScopedAttributeInterface::SCOPE_GLOBAL,
            'sort_order' => 20
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'gift_card_amounts', array_merge($this->getDefaultOptions(), [
            'label'      => 'Gift Card Amounts',
            'type'       => 'varchar',
            'input'      => 'text',
            'backend'    => 'Mageplaza\GiftCard\Model\Attribute\Backend\Amount',
            'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
            'sort_order' => 30
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'allow_amount_range', array_merge($this->getDefaultOptions(), [
            'label'      => 'Allow Amount Range',
            'type'       => 'int',
            'input'      => 'select',
            'source'     => 'Magento\Catalog\Model\Product\Attribute\Source\Boolean',
            'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
            'sort_order' => 40
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'min_amount', array_merge($this->getDefaultOptions(), [
            'label'      => 'Min Amount',
            'type'       => 'decimal',
            'input'      => 'price',
            'backend'    => 'Magento\Catalog\Model\Product\Attribute\Backend\Price',
            'class'      => 'validate-number',
            'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
            'sort_order' => 50
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'max_amount', array_merge($this->getDefaultOptions(), [
            'label'      => 'Max Amount',
            'type'       => 'decimal',
            'input'      => 'price',
            'backend'    => 'Magento\Catalog\Model\Product\Attribute\Backend\Price',
            'class'      => 'validate-number',
            'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
            'sort_order' => 60
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'price_rate', array_merge($this->getDefaultOptions(), [
            'label'      => 'Price Percentage',
            'type'       => 'decimal',
            'input'      => 'text',
            'class'      => 'validate-number',
            'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
            'sort_order' => 65
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'can_redeem', array_merge($this->getDefaultOptions(), [
            'label'      => 'Can Redeem',
            'type'       => 'int',
            'input'      => 'select',
            'source'     => 'Magento\Catalog\Model\Product\Attribute\Source\Boolean',
            'backend'    => 'Magento\Catalog\Model\Product\Attribute\Backend\Boolean',
            'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
            'sort_order' => 80
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'expire_after_day', array_merge($this->getDefaultOptions(), [
            'label'      => 'Expire After (days)',
            'type'       => 'varchar',
            'input'      => 'text',
            'backend'    => 'Mageplaza\GiftCard\Model\Attribute\Backend\Pattern',
            'global'     => ScopedAttributeInterface::SCOPE_WEBSITE,
            'sort_order' => 90
        ]));
        $catalogSetup->addAttribute(Product::ENTITY, 'gift_product_template', array_merge($this->getDefaultOptions(), [
            'label'      => 'Template',
            'type'       => 'text',
            'input'      => 'multiselect',
            'source'     => 'Mageplaza\GiftCard\Model\GiftCard\Template',
            'backend'    => 'Mageplaza\GiftCard\Model\Attribute\Backend\MultiSelect',
            'sort_order' => 100
        ]));

        $this->addRemoveApply($catalogSetup);

        /** Add Gift Product head block */
        $block = $this->blockFactory->create()
            ->load('gift_card_products', 'identifier');
        if (!$block->getId()) {
            $headBlock = [
                'title'      => 'Gift Card Product',
                'identifier' => 'gift_card_products',
                'stores'     => [0],
                'is_active'  => 1,
                'content'    => '<p><img src="{{media url="mageplaza/default/gift_product_banner.jpg"}}" alt="Gift Product Banner" width="100%" /></p>'
            ];
            $block->setData($headBlock)
                ->save();
        }

        /** Add sample template data */
        $sampleTemplates = [
            [
                'name'   => 'Amazon Style',
                'status' => 1,
                'title'  => 'Gift Card',
                'design' => '{"giftcard":{"label":"Gift Card","width":500,"height":530,"css":{"border-radius":"5px"}},"image":{"label":"Image","sampleContent":"Image","width":500,"height":300,"top":0,"left":0},"message":{"label":"Message","sampleContent":"Hope you enjoy this gift card!","width":500,"height":100,"top":300,"left":0,"css":{"border-bottom":"1px solid #ccc","font-size":"15px","color":"#000","padding":"25px"}},"value":{"label":"Value","sampleContent":"$100.00","width":176,"height":50,"top":410,"left":30,"css":{"font-size":"40px","padding":"5px","color":"rgb(68, 68, 68)"}},"title":{"label":"Title","sampleContent":"Gift Card","width":104,"height":40,"top":480,"left":40,"css":{"font-size":"18px","color":"#868686"}},"logo":{"label":"Logo","sampleContent":"Logo","width":150,"height":100,"top":410,"left":320}}'
            ]
        ];
        $setup->getConnection()->insertMultiple($installer->getTable('mageplaza_giftcard_template'), $sampleTemplates);

        $installer->endSetup();
    }

    /**
     * Add/remove field from gift card product
     *
     * @param \Magento\Catalog\Setup\CategorySetup $catalogSetup
     *
     * @return $this
     */
    protected function addRemoveApply($catalogSetup)
    {
        $fieldAdd = ['weight'];
        foreach ($fieldAdd as $field) {
            $applyTo = $catalogSetup->getAttribute('catalog_product', $field, 'apply_to');
            if ($applyTo) {
                $applyTo = explode(',', $applyTo);
                if (!in_array(GiftCard::TYPE_GIFTCARD, $applyTo)) {
                    $applyTo[] = GiftCard::TYPE_GIFTCARD;
                    $catalogSetup->updateAttribute('catalog_product', 'weight', 'apply_to', join(',', $applyTo));
                }
            }
        }

        $fieldRemove = ['cost'];
        foreach ($fieldRemove as $field) {
            $applyTo = explode(',', $catalogSetup->getAttribute('catalog_product', $field, 'apply_to'));
            if (in_array(GiftCard::TYPE_GIFTCARD, $applyTo)) {
                foreach ($applyTo as $k => $v) {
                    if ($v == GiftCard::TYPE_GIFTCARD) {
                        unset($applyTo[$k]);
                        break;
                    }
                }
                $catalogSetup->updateAttribute('catalog_product', $field, 'apply_to', join(',', $applyTo));
            }
        }

        return $this;
    }

    /**
     * @return array
     */
    protected function getDefaultOptions()
    {
        return [
            'group'                   => 'Gift Card Information',
            'backend'                 => '',
            'frontend'                => '',
            'class'                   => '',
            'source'                  => '',
            'global'                  => ScopedAttributeInterface::SCOPE_STORE,
            'visible'                 => true,
            'required'                => false,
            'user_defined'            => true,
            'default'                 => '',
            'searchable'              => false,
            'filterable'              => false,
            'comparable'              => false,
            'visible_on_front'        => false,
            'unique'                  => false,
            'apply_to'                => GiftCard::TYPE_GIFTCARD,
            'used_in_product_listing' => true
        ];
    }
}
