<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_GiftCard
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\GiftCard\Observer;

use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Mageplaza\GiftCard\Helper\Data as DataHelper;
use Mageplaza\GiftCard\Model\GiftCardFactory;
use Mageplaza\GiftCard\Model\Transaction\Action;
use Mageplaza\GiftCard\Model\TransactionFactory;

/**
 * Class SalesConvertQuote
 * @package Mageplaza\GiftCard\Observer
 */
class SalesConvertQuote implements ObserverInterface
{
    /**
     * @var \Mageplaza\GiftCard\Model\GiftCardFactory
     */
    protected $giftCardFactory;

    /**
     * @var \Mageplaza\GiftCard\Model\TransactionFactory
     */
    protected $transactionFactory;

    /**
     * @var \Mageplaza\GiftCard\Helper\Data
     */
    protected $_helper;

    /**
     * SalesConvertQuote constructor.
     *
     * @param \Mageplaza\GiftCard\Model\GiftCardFactory $giftCardFactory
     * @param \Mageplaza\GiftCard\Model\TransactionFactory $transactionFactory
     * @param \Mageplaza\GiftCard\Helper\Data $helper
     */
    public function __construct(GiftCardFactory $giftCardFactory, TransactionFactory $transactionFactory, DataHelper $helper)
    {
        $this->giftCardFactory    = $giftCardFactory;
        $this->transactionFactory = $transactionFactory;
        $this->_helper            = $helper;
    }

    /**
     * @param Observer $observer
     * @return $this|void
     */
    public function execute(Observer $observer)
    {
        /** @var \Magento\Sales\Model\Order $order */
        $order = $observer->getEvent()->getOrder();

        /** @var \Magento\Quote\Model\Quote $quote */
        $quote   = $observer->getEvent()->getQuote();
        $address = $quote->isVirtual() ? $quote->getBillingAddress() : $quote->getShippingAddress();

        $giftCardsUsed = $quote->getGiftCards();
        if ($giftCardsUsed) {
            $giftCards = DataHelper::jsonDecode($giftCardsUsed);
            foreach ($giftCards as $code => $amount) {
                $this->giftCardFactory->create()
                    ->loadByCode($code)
                    ->spentForOrder($amount, $order);
            }

            $order->setGiftCards($giftCardsUsed);

            $order->setGiftCardAmount($address->getGiftCardAmount());
            $order->setBaseGiftCardAmount($address->getBaseGiftCardAmount());
        }

        $baseCreditAmount = $address->getBaseGiftCreditAmount();
        if (abs($baseCreditAmount) > 0.0001) {
            $order->setGiftCreditAmount($address->getGiftCreditAmount());
            $order->setBaseGiftCreditAmount($address->getBaseGiftCreditAmount());

            $this->transactionFactory->create()
                ->createTransaction(
                    Action::ACTION_SPEND,
                    $baseCreditAmount,
                    $order->getCustomerId(),
                    ['order_increment_id' => $order->getIncrementId()]
                );
        }
        $this->_helper->getCheckoutSession()->setGiftCardsData([]);

        return $this;
    }
}
