<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_GiftCard
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\GiftCard\Observer;

use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Mageplaza\GiftCard\Helper\Product;
use Mageplaza\GiftCard\Model\GiftCard\Action;
use Mageplaza\GiftCard\Model\GiftCardFactory;
use Mageplaza\GiftCard\Model\Product\Type\GiftCard;
use Mageplaza\GiftCard\Model\TransactionFactory;
use Psr\Log\LoggerInterface;

/**
 * Class CreditmemoSaveAfter
 * @package Mageplaza\GiftCard\Observer
 */
class CreditmemoSaveAfter implements ObserverInterface
{
    /**
     * @var \Mageplaza\GiftCard\Helper\Product
     */
    protected $_productHelper;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @var \Mageplaza\GiftCard\Model\GiftCardFactory
     */
    protected $giftCardFactory;

    /**
     * @var \Mageplaza\GiftCard\Model\TransactionFactory
     */
    protected $transactionFactory;

    /**
     * CreditmemoSaveAfter constructor.
     *
     * @param \Mageplaza\GiftCard\Helper\Product $productHelper
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Mageplaza\GiftCard\Model\GiftCardFactory $giftCardFactory
     * @param \Mageplaza\GiftCard\Model\TransactionFactory $transactionFactory
     */
    public function __construct(
        Product $productHelper,
        LoggerInterface $logger,
        GiftCardFactory $giftCardFactory,
        TransactionFactory $transactionFactory
    )
    {
        $this->_productHelper     = $productHelper;
        $this->logger             = $logger;
        $this->giftCardFactory    = $giftCardFactory;
        $this->transactionFactory = $transactionFactory;
    }

    /**
     * @param \Magento\Framework\Event\Observer $observer
     *
     * @return $this
     */
    public function execute(Observer $observer)
    {
        /** @var \Magento\Sales\Model\Order\Creditmemo $creditmemo */
        $creditmemo = $observer->getEvent()->getCreditmemo();
        if (!$creditmemo->getRefundGiftCardFlag()) {
            return $this;
        }

        /** @var \Magento\Sales\Model\Order\Creditmemo\Item $item */
        foreach ($creditmemo->getAllItems() as $item) {
            /** @var \Magento\Sales\Model\Order\Item $orderItem */
            $orderItem = $item->getOrderItem();
            if ($orderItem->isDummy() || ($orderItem->getProductType() != GiftCard::TYPE_GIFTCARD)) {
                continue;
            }

            $this->_productHelper->refundGiftCode($orderItem, $item->getQty());
        }

        /** @var \Magento\Sales\Model\Order $order */
        $order = $creditmemo->getOrder();

        if ($this->_productHelper->allowRefundGiftCard()) {
            /** Refund Gift Cards */
            $giftCards = $creditmemo->getGiftCards() ?: [];
            foreach ($giftCards as $code => $amount) {
                try {
                    $giftCard = $this->giftCardFactory->create()
                        ->loadByCode($code);
                    if ($giftCard->getId()) {
                        $giftCard->addBalance($amount)
                            ->setAction(Action::ACTION_REFUND)
                            ->setActionVars(['order_increment_id' => $order->getIncrementId()])
                            ->save();
                    }
                } catch (\Exception $e) {
                    $this->logger->critical($e->getMessage());
                }
            }
        }

        /** Refund Gift Credit */
        $giftCredit = $creditmemo->getBaseGiftCreditAmount();
        if (abs($giftCredit) > 0.0001) {
            try {
                $this->transactionFactory->create()
                    ->createTransaction(
                        \Mageplaza\GiftCard\Model\Transaction\Action::ACTION_REFUND,
                        abs($giftCredit),
                        $order->getCustomerId(),
                        ['order_increment_id' => $order->getIncrementId()]
                    );
            } catch (\Exception $e) {
                $this->logger->critical($e->getMessage());
            }
        }

        return $this;
    }
}
