<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_GiftCard
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\GiftCard\Observer;

use Magento\Framework\App\ActionInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Escaper;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Message\ManagerInterface;
use Magento\Framework\UrlInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Mageplaza\GiftCard\Helper\Checkout as DataHelper;
use Mageplaza\GiftCard\Model\GiftCardFactory;

/**
 * Class CouponPost
 * @package Mageplaza\GiftCard\Observer
 */
class CouponPost implements ObserverInterface
{
    /**
     * @var \Magento\Framework\UrlInterface
     */
    protected $_url;

    /**
     * @var \Magento\Framework\Escaper
     */
    protected $escaper;

    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    protected $messageManager;

    /**
     * @type \Mageplaza\GiftCard\Model\GiftCardFactory
     */
    protected $_giftcardFactory;

    /**
     * @type \Mageplaza\GiftCard\Helper\Checkout
     */
    protected $_dataHelper;

    /**
     * @var \Magento\Quote\Api\CartRepositoryInterface
     */
    protected $quoteRepository;

    /**
     * CouponPost constructor.
     *
     * @param \Magento\Framework\UrlInterface $url
     * @param \Magento\Framework\Escaper $escaper
     * @param \Magento\Framework\Message\ManagerInterface $managerInterface
     * @param \Mageplaza\GiftCard\Model\GiftCardFactory $giftcardFactory
     * @param \Mageplaza\GiftCard\Helper\Checkout $dataHelper
     * @param \Magento\Quote\Api\CartRepositoryInterface $quoteRepository
     */
    function __construct(
        UrlInterface $url,
        Escaper $escaper,
        ManagerInterface $managerInterface,
        GiftCardFactory $giftcardFactory,
        DataHelper $dataHelper,
        CartRepositoryInterface $quoteRepository
    )
    {
        $this->_url             = $url;
        $this->escaper          = $escaper;
        $this->messageManager   = $managerInterface;
        $this->_giftcardFactory = $giftcardFactory;
        $this->_dataHelper      = $dataHelper;
        $this->quoteRepository  = $quoteRepository;
    }

    /**
     * Execute
     *
     * @param \Magento\Framework\Event\Observer $observer
     *
     * @return $this|\Magento\Framework\Controller\Result\Redirect
     */
    public function execute(Observer $observer)
    {
        if (!$this->_dataHelper->isEnabled() || !$this->_dataHelper->isUsedCouponBox()) {
            return $this;
        }

        /** @var \Magento\Checkout\Controller\Cart\CouponPost $action */
        $action = $observer->getEvent()->getControllerAction();

        /** @type \Magento\Framework\App\RequestInterface $request */
        $request    = $observer->getEvent()->getRequest();
        $couponCode = ($request->getParam('remove') == 1) ? '' : trim($request->getParam('coupon_code'));

        if (!strlen($couponCode)) {
            /** @var \Magento\Quote\Model\Quote $quote */
            $quote = $this->_dataHelper->getCheckoutSession()->getQuote();
            if ($quote->getCouponCode()) {
                return $this;
            }

            $giftCards = $this->_dataHelper->getGiftCardsUsed($quote);
            if ($giftCards && sizeof($giftCards)) {
                $this->_dataHelper->removeGiftCard(null, true);
                $this->messageManager->addSuccessMessage(__('You canceled the gift card code.'));

                return $this->_goBack($action);
            }

            return $this;
        }

        if ($this->_dataHelper->canUsedGiftCard()) {
            try {
                $this->_dataHelper->addGiftCards($couponCode);
                $this->messageManager->addSuccessMessage(__('You used gift card code "%1".', $this->escaper->escapeHtml($couponCode)));

                return $this->_goBack($action);
            } catch (\Exception $e) {
                ObjectManager::getInstance()->get('Psr\Log\LoggerInterface')->critical($e);
            }
        }

        return $this;
    }

    /**
     * @param \Magento\Checkout\Controller\Cart\CouponPost $action
     *
     * @return $this
     */
    protected function _goBack($action)
    {
        $action->getActionFlag()->set('', ActionInterface::FLAG_NO_DISPATCH, true);
        $action->getResponse()->setRedirect($this->_url->getUrl('checkout/cart'));

        return $this;
    }
}
