<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_GiftCard
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\GiftCard\Model\ResourceModel;

use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Store\Model\StoreManagerInterface;
use Mageplaza\GiftCard\Helper\Data;

/**
 * Class GiftCard
 * @package Mageplaza\GiftCard\Model\ResourceModel
 */
class GiftCard extends AbstractDb
{
    /**
     * @type \Magento\Store\Model\StoreManagerInterface
     */
    protected $_storeManager;

    /**
     * Constructor
     *
     * @param \Magento\Framework\Model\ResourceModel\Db\Context $context
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param null $connectionName
     */
    public function __construct(
        Context $context,
        StoreManagerInterface $storeManager,
        $connectionName = null
    )
    {
        $this->_storeManager = $storeManager;

        parent::__construct($context, $connectionName);
    }

    /**
     * Constructor
     */
    protected function _construct()
    {
        $this->_init('mageplaza_giftcard', 'giftcard_id');
    }

    /**
     * @param \Magento\Framework\Model\AbstractModel $object
     *
     * @return $this
     */
    protected function _beforeSave(AbstractModel $object)
    {
        parent::_beforeSave($object);

        $templateFields = $object->getTemplateFields();
        $extraContent   = $object->getExtraContent();

        if (is_array($templateFields)) {
            $object->setTemplateFields(Data::jsonEncode($templateFields));
        }

        if (is_array($extraContent)) {
            $object->setExtraContent(Data::jsonEncode($extraContent));
        }

        return $this;
    }

    /**
     * Check for unique values existence
     *
     * @param $object
     * @param $code
     *
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function checkCodeAvailable($object, $code)
    {
        $select = $this->getConnection()->select()->from($this->getMainTable());
        $select->where('code=?', trim($code));
        if ($object->getId() || (string)$object->getId() === '0') {
            $select->where($this->getIdFieldName() . '!=?', $object->getId());
        }
        $test = $this->getConnection()->fetchRow($select);

        return (boolean)$test;
    }

    /**
     * @param $ids
     * @param $status
     *
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function updateStatuses($ids, $status)
    {
        $this->getConnection()->update(
            $this->getMainTable(),
            ['status' => $status],
            [$this->getIdFieldName() . ' IN (?)' => $ids]
        );

        return $this;
    }

    /**
     * Perform actions after object load
     *
     * @param \Magento\Framework\Model\AbstractModel|\Magento\Framework\DataObject $object
     *
     * @return $this
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    protected function _afterLoad(AbstractModel $object)
    {
        parent::_afterLoad($object);

        $templateFields = $object->getTemplateFields();
        $extraContent   = $object->getExtraContent();

        if ($templateFields) {
            $object->addData(Data::jsonDecode($templateFields));
        }

        if ($extraContent) {
            $object->setExtraContent(Data::jsonDecode($extraContent));
        }

        return $this;
    }

    /**
     * @param $giftCard
     * @param $qty
     *
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function createMultiple($giftCard, $qty)
    {
        $resultData = [];
        $data       = [
            'init_balance'    => $giftCard->getBalance(),
            'balance'         => $giftCard->getBalance(),
            'status'          => $giftCard->getStatus(),
            'can_redeem'      => $giftCard->getCanRedeem(),
            'store_id'        => $giftCard->getStoreId(),
            'pool_id'         => $giftCard->getPoolId(),
            'template_id'     => $giftCard->getTemplateId(),
            'image'           => $giftCard->getImage(),
            'template_fields' => $giftCard->getTemplateFields(),
            'expired_at'      => $giftCard->getExpiredAt()
        ];

        while ($qty--) {
            $data['code'] = $giftCard->generateCode($giftCard->getPattern());
            $resultData[] = $data;
        }

        $this->getConnection()->insertMultiple($this->getMainTable(), $resultData);

        return array_column($resultData, 'code');
    }

    /**
     * @param $data
     * @param $where
     *
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function updateMultiple($data, $where)
    {
        $this->getConnection()->update($this->getMainTable(), $data, $where);

        return $this;
    }

    /**
     * Generate multi Gift Card transaction
     *
     * @param array $objects
     *
     * @return $this
     * @throws \Exception
     */
    function generateMultiGiftCard($objects)
    {
        $this->beginTransaction();

        try {
            /** @type \Mageplaza\GiftCard\Model\GiftCard $object */
            foreach ($objects as $object) {
                $object->save();
            }

            $this->commit();
        } catch (\Exception $e) {
            $this->rollBack();
            throw $e;
        }

        return $this;
    }
}
