<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_GiftCard
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\GiftCard\Model;

use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\DataObject\IdentityInterface;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Mageplaza\GiftCard\Helper\Data as DataHelper;
use Mageplaza\GiftCard\Model\Source\Status;

/**
 * Class Pool
 * @package Mageplaza\GiftCard\Model
 */
class Pool extends AbstractModel implements IdentityInterface
{
    /**
     * Cache tag
     */
    const CACHE_TAG = 'mageplaza_giftcard_pool';

    /**
     * Prefix of model events names
     *
     * @var string
     */
    protected $_eventPrefix = 'mageplaza_giftcard_pool';

    /**
     * @var \Mageplaza\GiftCard\Helper\Data
     */
    protected $_helper;

    /**
     * @var \Mageplaza\GiftCard\Model\GiftCardFactory
     */
    protected $giftCardFactory;

    /**
     * Pool constructor.
     *
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Mageplaza\GiftCard\Helper\Data $helper
     * @param \Mageplaza\GiftCard\Model\GiftCardFactory $giftCardFactory
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource|null $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        Context $context,
        Registry $registry,
        DataHelper $helper,
        GiftCardFactory $giftCardFactory,
        AbstractResource $resource = null,
        AbstractDb $resourceCollection = null,
        array $data = []
    )
    {
        $this->_helper         = $helper;
        $this->giftCardFactory = $giftCardFactory;

        parent::__construct($context, $registry, $resource, $resourceCollection, $data);
    }

    /**
     * Constructor
     */
    protected function _construct()
    {
        parent::_construct();
        $this->_init('Mageplaza\GiftCard\Model\ResourceModel\Pool');
    }

    /**
     * @return array
     */
    public function getIdentities()
    {
        return [self::CACHE_TAG . '_' . $this->getId()];
    }

    /**
     * @inheritdoc
     */
    public function beforeSave()
    {
        parent::beforeSave();

        if ($this->getId() && $this->getCanInherit()) {
            $dataUpdate    = [
                'can_redeem'      => $this->getCanRedeem(),
                'store_id'        => $this->getStoreId(),
                'template_id'     => $this->getTemplateId(),
                'image'           => $this->getImage(),
                'expired_at'      => $this->getExpiredAt()
                    ? date('Y-m-d', strtotime($this->getExpiredAt()))
                    : null,
                'template_fields' => $this->getTemplateFields()
            ];
            $balanceChange = $this->getBalance() - $this->getOrigData('balance');
            if ($balanceChange != 0) {
                $sqlVar                = '(`balance` + ' . $balanceChange . ')';
                $dataUpdate['balance'] = new \Zend_Db_Expr("(CASE WHEN ({$sqlVar} < 0) THEN 0 ELSE {$sqlVar} END)");
            }

            $this->giftCardFactory->create()->updateMultiple($dataUpdate, ['pool_id = ?' => $this->getId()]);
        }

        return $this;
    }

    /**
     * @return bool
     */
    public function isActive()
    {
        return $this->getStatus() == Status::STATUS_ACTIVE;
    }

    /**
     * Get gift card status label
     *
     * @param null $status
     *
     * @return \Magento\Framework\Phrase
     */
    public function getStatusLabel($status = null)
    {
        if (is_null($status)) {
            $status = $this->getStatus();
        }

        $allStatus = Status::getOptionArray();

        return isset($allStatus[$status]) ? $allStatus[$status] : __('Undefined');
    }
}
