<?php
/**
 * Mageplaza
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Mageplaza.com license that is
 * available through the world-wide-web at this URL:
 * https://www.mageplaza.com/LICENSE.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Mageplaza
 * @package     Mageplaza_GiftCard
 * @copyright   Copyright (c) Mageplaza (https://www.mageplaza.com/)
 * @license     https://www.mageplaza.com/LICENSE.txt
 */

namespace Mageplaza\GiftCard\Block\Adminhtml\Pool\Edit\Tab;

use Magento\Backend\Block\Template\Context;
use Magento\Backend\Block\Widget\Form\Generic;
use Magento\Backend\Block\Widget\Tab\TabInterface;
use Magento\Config\Model\Config\Source\Yesno;
use Magento\Framework\Data\FormFactory;
use Magento\Framework\Registry;
use Magento\Store\Model\System\Store;
use Mageplaza\GiftCard\Helper\Data as DataHelper;
use Mageplaza\GiftCard\Model\Source\Status;

/**
 * Class Information
 * @package Mageplaza\GiftCard\Block\Adminhtml\Pool\Edit\Tab
 */
class Information extends Generic implements TabInterface
{
    /** @var \Magento\Store\Model\System\Store */
    protected $_systemStore;

    /** @var \Magento\Config\Model\Config\Source\Yesno */
    protected $_yesnoOptions;

    /** @var \Mageplaza\GiftCard\Helper\Data */
    protected $_helper;

    /**
     * @var \Mageplaza\GiftCard\Model\Source\Status
     */
    protected $_status;

    /**
     * Information constructor.
     *
     * @param \Magento\Backend\Block\Template\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Data\FormFactory $formFactory
     * @param \Magento\Store\Model\System\Store $systemStore
     * @param \Magento\Config\Model\Config\Source\Yesno $yesnoOptions
     * @param \Mageplaza\GiftCard\Helper\Data $helper
     * @param \Mageplaza\GiftCard\Model\Source\Status $statusOptions
     * @param array $data
     */
    public function __construct(
        Context $context,
        Registry $registry,
        FormFactory $formFactory,
        Store $systemStore,
        Yesno $yesnoOptions,
        DataHelper $helper,
        Status $statusOptions,
        array $data = []
    )
    {
        $this->_systemStore  = $systemStore;
        $this->_yesnoOptions = $yesnoOptions;
        $this->_helper       = $helper;
        $this->_status       = $statusOptions;

        parent::__construct($context, $registry, $formFactory, $data);
    }

    /**
     * @inheritdoc
     */
    protected function _prepareForm()
    {
        /* @var $model \Mageplaza\GiftCard\Model\Pool */
        $model = $this->_coreRegistry->registry('current_pool');

        /** @var \Magento\Framework\Data\Form $form */
        $form = $this->_formFactory->create();

        $informationFieldset = $form->addFieldset('information_fieldset', ['legend' => __('Pool Information')]);

        if ($model->getId()) {
            $informationFieldset->addField('pool_id', 'hidden', ['name' => 'id']);
        }

        $informationFieldset->addField('name', 'text', [
            'name'     => 'name',
            'label'    => __('Name'),
            'title'    => __('Name'),
            'required' => true
        ]);
        $informationFieldset->addField('status', 'select', [
            'name'   => 'status',
            'label'  => __('Status'),
            'title'  => __('Status'),
            'values' => $this->_status->toOptionArray(),
            'note'   => 'The gift cards of this pool cannot be used if the pool status is inactive.'
        ]);

        $informationFieldset->addField('can_inherit', 'select', [
            'name'   => 'can_inherit',
            'label'  => __('Can Inherit'),
            'title'  => __('Can Inherit'),
            'values' => $this->_yesnoOptions->toOptionArray(),
            'note'   => 'If yes, when you change this pool information, the gift card information will be changed. Except: Status attribute'
        ]);

        $giftCardFieldset = $form->addFieldset('gift_card_fieldset', ['legend' => __('Gift Card Information')]);

        $giftCardFieldset->addType('price', 'Magento\Catalog\Block\Adminhtml\Product\Helper\Form\Price');
        $giftCardFieldset->addField('balance', 'price', [
            'name'     => 'balance',
            'label'    => __('Balance'),
            'title'    => __('Balance'),
            'class'    => 'validate-greater-than-zero',
            'required' => true
        ])->setAfterElementHtml('$');

        if ($this->_helper->getGeneralConfig('enable_credit')) {
            $giftCardFieldset->addField('can_redeem', 'select', [
                'name'   => 'can_redeem',
                'label'  => __('Can Redeem'),
                'title'  => __('Can Redeem'),
                'values' => $this->_yesnoOptions->toOptionArray()
            ]);
            if (!$model->getId()) {
                $model->setData('can_redeem', $this->_helper->getGeneralConfig('can_redeem'));
            }
        }

        if (!$this->_storeManager->isSingleStoreMode()) {
            /** @var \Magento\Framework\Data\Form\Element\Renderer\RendererInterface $rendererBlock */
            $rendererBlock = $this->getLayout()->createBlock('Magento\Backend\Block\Store\Switcher\Form\Renderer\Fieldset\Element');
            $giftCardFieldset->addField('store_id', 'select', [
                'name'   => 'store_id',
                'label'  => __('Store'),
                'title'  => __('Store'),
                'values' => $this->_systemStore->getStoreValuesForForm()
            ])->setRenderer($rendererBlock);
        }

        $giftCardFieldset->addField('expired_at', 'date', [
            'name'        => 'expired_at',
            'label'       => __('Expires At'),
            'title'       => __('Expires At'),
            'class'       => 'validate-date',
            'date_format' => $this->_localeDate->getDateFormat(\IntlDateFormatter::MEDIUM)
        ]);

        /** @var \Magento\Backend\Block\Widget\Form\Element\Dependence $dependencies */
        $dependencies = $this->getLayout()->createBlock('Magento\Backend\Block\Widget\Form\Element\Dependence');

        /** Add template fields: image, sender name, recipient name, message */
        $this->_helper->getTemplateHelper()
            ->getTemplateFieldSet($model, $giftCardFieldset, $dependencies);

        // define field dependencies
        $this->setChild('form_after', $dependencies);

        $form->setValues($model->getData());
        $this->setForm($form);

        return parent::_prepareForm();
    }

    /**
     * Prepare label for tab
     *
     * @return string
     */
    public function getTabLabel()
    {
        return __('Gift Code Information');
    }

    /**
     * Prepare title for tab
     *
     * @return string
     */
    public function getTabTitle()
    {
        return __('Gift Code Information');
    }

    /**
     * {@inheritdoc}
     */
    public function canShowTab()
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function isHidden()
    {
        return false;
    }
}
