<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\Cms\Model\Page;

use Magento\Cms\Model\Page;
use Magento\Cms\Model\ResourceModel\Page\CollectionFactory;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\Request\DataPersistorInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\AuthorizationInterface;

/**
 * Class DataProvider
 */
class DataProvider extends \Magento\Ui\DataProvider\AbstractDataProvider
{
    /**
     * @var \Magento\Cms\Model\ResourceModel\Page\Collection
     */
    protected $collection;

    /**
     * @var DataPersistorInterface
     */
    protected $dataPersistor;

    /**
     * @var array
     */
    protected $loadedData;

    /**
     * @var AuthorizationInterface
     */
    private $authorization;

    /**
     * @var RequestInterface
     */
    private $request;

    /**
     * @var CustomLayoutManagerInterface
     */
    private $customLayoutManager;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @param string $name
     * @param string $primaryFieldName
     * @param string $requestFieldName
     * @param CollectionFactory $pageCollectionFactory
     * @param DataPersistorInterface $dataPersistor
     * @param array $meta
     * @param array $data
     * @param AuthorizationInterface|null $authorization
     * @param RequestInterface|null $request
     * @param CustomLayoutManagerInterface|null $customLayoutManager
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        $name,
        $primaryFieldName,
        $requestFieldName,
        CollectionFactory $pageCollectionFactory,
        DataPersistorInterface $dataPersistor,
        array $meta = [],
        array $data = [],
        AuthorizationInterface $authorization = null,
        RequestInterface $request = null,
        CustomLayoutManagerInterface $customLayoutManager = null
    ) {
        $this->collection = $pageCollectionFactory->create();
        $this->collectionFactory = $pageCollectionFactory;
        $this->dataPersistor = $dataPersistor;
        parent::__construct($name, $primaryFieldName, $requestFieldName, $meta, $data);
        $this->authorization = $authorization ?? ObjectManager::getInstance()->get(AuthorizationInterface::class);
        $this->meta = $this->prepareMeta($this->meta);
        $this->request = $request ?? ObjectManager::getInstance()->get(RequestInterface::class);
        $this->customLayoutManager = $customLayoutManager
            ?? ObjectManager::getInstance()->get(CustomLayoutManagerInterface::class);
    }

    /**
     * Find requested page.
     *
     * @return Page|null
     */
    private function findCurrentPage()
    {
        if ($this->getRequestFieldName() && ($pageId = (int)$this->request->getParam($this->getRequestFieldName()))) {
            //Loading data for the collection.
            $this->getData();
            return $this->collection->getItemById($pageId);
        }

        return null;
    }

    /**
     * Prepares Meta
     *
     * @param array $meta
     * @return array
     */
    public function prepareMeta(array $meta)
    {
        return $meta;
    }

    /**
     * Get data
     *
     * @return array
     */
    public function getData()
    {
        if (isset($this->loadedData)) {
            return $this->loadedData;
        }
        $this->collection = $this->collectionFactory->create();
        $items = $this->collection->getItems();
        /** @var $page \Magento\Cms\Model\Page */
        foreach ($items as $page) {
            $this->loadedData[$page->getId()] = $page->getData();
            if ($page->getCustomLayoutUpdateXml() || $page->getLayoutUpdateXml()) {
                //Deprecated layout update exists.
                $this->loadedData[$page->getId()]['layout_update_selected'] = '_existing_';
            }
        }

        $data = $this->dataPersistor->get('cms_page');
        if (!empty($data)) {
            $page = $this->collection->getNewEmptyItem();
            $page->setData($data);
            $this->loadedData[$page->getId()] = $page->getData();
            if ($page->getCustomLayoutUpdateXml() || $page->getLayoutUpdateXml()) {
                $this->loadedData[$page->getId()]['layout_update_selected'] = '_existing_';
            }
            $this->dataPersistor->clear('cms_page');
        }

        return $this->loadedData;
    }

    /**
     * @inheritDoc
     */
    public function getMeta()
    {
        $meta = parent::getMeta();

        if (!$this->authorization->isAllowed('Magento_Cms::save_design')) {
            $designMeta = [];
            $designFieldSets = ['design', 'custom_design_update'];

            foreach ($designFieldSets as $fieldSet) {
                $designMeta[$fieldSet] = [
                    'arguments' => [
                        'data' => [
                            'config' => [
                                'disabled' => true,
                            ],
                        ],
                    ],
                ];
            }

            $meta = array_merge_recursive($meta, $designMeta);
        }

        //List of custom layout files available for current page.
        $options = [['label' => 'No update', 'value' => '_no_update_']];
        if ($page = $this->findCurrentPage()) {
            //We must have a specific page selected.
            //If custom layout XML is set then displaying this special option.
            if ($page->getCustomLayoutUpdateXml() || $page->getLayoutUpdateXml()) {
                $options[] = ['label' => 'Use existing layout update XML', 'value' => '_existing_'];
            }
            foreach ($this->customLayoutManager->fetchAvailableFiles($page) as $layoutFile) {
                $options[] = ['label' => $layoutFile, 'value' => $layoutFile];
            }
        }
        $customLayoutMeta = [
            'design' => [
                'children' => [
                    'custom_layout_update_select' => [
                        'arguments' => [
                            'data' => ['options' => $options]
                        ]
                    ]
                ]
            ]
        ];
        $meta = array_merge_recursive($meta, $customLayoutMeta);

        return $meta;
    }
}
