<?php
/**
 * File: VarnishUrlRegenerator.php
 *
 * @author Maciej Sławik <maciej.slawik@lizardmedia.pl>
 * @copyright Copyright (C) 2018 Lizard Media (http://lizardmedia.pl)
 */

namespace LizardMedia\VarnishWarmer\Model\QueueHandler;

use cURL\Event;
use cURL\Request;
use LizardMedia\VarnishWarmer\Api\QueueHandler\VarnishUrlRegeneratorInterface;

/**
 * Class VarnishUrlRegenerator
 * @package LizardMedia\VarnishWarmer\Model\QueueHandler
 */
class VarnishUrlRegenerator extends AbstractQueueHandler implements VarnishUrlRegeneratorInterface
{
    const PROCESS_TYPE = 'REGENERATE';

    /**
     * @param string $url
     * @return void
     */
    public function addUrlToRegenerate(string $url): void
    {
        $this->requests[] = new Request($url);
        $this->total++;
    }

    /**
     * @return void
     */
    public function runRegenerationQueue(): void
    {
        $this->buildQueue();
        $this->runQueueProcesses();
    }

    /**
     * @return void
     */
    protected function buildQueue(): void
    {

        $this->queue
            ->getDefaultOptions()
            ->set(CURLOPT_TIMEOUT, self::CURL_TIMEOUT)
            ->set(CURLOPT_USERAGENT, 'LizardMedia_VarnishWarmer')
            ->set(CURLOPT_RETURNTRANSFER, true);

        $requests = &$this->requests;

        $this->queue->addListener('complete', function (Event $event) use (&$requests) {
            $this->counter++;
            $info = curl_getinfo($event->request->getHandle());

//            echo PHP_EOL . '<pre>';
//            print_r($info);
//            echo '</pre>' . PHP_EOL;

            $responseHeader = get_headers($info['url']);
            $age = $this->getHeaderAge($responseHeader);
            if($age < 5) {
                usleep($this->getRegenerationSleepMicroseconds());
            }

            if($info['http_code'] !== 200) {
                $this->log('ERROR: ' . json_encode($info) . ' - ' . json_encode($responseHeader));
            } else {
                $this->log($info['url'] . ' - Age: ' . $age);
            }

            $this->logProgress();
            if ($next = array_pop($requests)) {
                $event->queue->attach($next);
            }
        });
    }

    private function getHeaderAge($headers) {
        $prefix = 'Age: ';
        $keyNumber = key(preg_grep("/\b$prefix\b/i", $headers));
        if(empty($keyNumber)) {
            $this->log('ERROR: Age not found for header ' . json_encode($headers));
            return 0;
        }

        return (int) str_replace($prefix, '', $headers[$keyNumber]);
    }

    /**
     * @return void
     */
    protected function runQueueProcesses(): void
    {
        $numberOfProcesses = $this->getNumberOfParallelProcesses();
        for ($i = 0; $i < $numberOfProcesses; $i++) {
            $this->queue->attach(array_pop($this->requests));
        }

        if ($this->queue->count() > 0) {
            $this->queue->send();
        }
    }

    /**
     * @return int
     */
    protected function getMaxNumberOfProcesses(): int
    {
        return $this->configProvider->getMaxConcurrentRegenerationProcesses();
    }

    /**
     * @return int
     */
    protected function getRegenerationSleepMicroseconds(): int
    {
        return $this->configProvider->getRegenerationSleepMicroseconds();
    }

    /**
     * @return string
     */
    protected function getQueueProcessType(): string
    {
        return self::PROCESS_TYPE;
    }
}
