<?php
/**
 * File: CacheCleaner.php
 *
 * @author Maciej Sławik <maciej.slawik@lizardmedia.pl>
 * @copyright Copyright (C) 2018 Lizard Media (http://lizardmedia.pl)
 */

namespace LizardMedia\VarnishWarmer\Helper;

use LizardMedia\VarnishWarmer\Api\Config\PurgingConfigProviderInterface;
use LizardMedia\VarnishWarmer\Api\LockHandler\LockInterface;
use LizardMedia\VarnishWarmer\Api\QueueHandler\VarnishUrlPurgerInterface;
use LizardMedia\VarnishWarmer\Api\QueueHandler\VarnishUrlRegeneratorInterface;
use LizardMedia\VarnishWarmer\Api\UrlProvider\CategoryUrlProviderInterface;
use LizardMedia\VarnishWarmer\Api\UrlProvider\ProductUrlProviderInterface;
use LizardMedia\VarnishWarmer\Model\UrlProvider\CmsPageUrlProvider;
use LizardMedia\VarnishWarmer\Model\QueueHandler\VarnishUrlRegeneratorFactory;
use LizardMedia\VarnishWarmer\Model\QueueHandler\VarnishUrlPurgerFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\Store;

//TODO remove this entire class and change it to a service contract
/**
 * Class CacheCleaner
 * @package LizardMedia\VarnishWarmer\Helper
 */
class CacheCleaner
{
    /**
     * @var VarnishUrlRegeneratorInterface
     */
    protected $varnishUrlRegenerator;

    /**
     * @var VarnishUrlPurgerInterface
     */
    protected $varnishUrlPurger;

    /**
     * @var LockInterface
     */
    protected $lockHandler;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var ProductUrlProviderInterface
     */
    protected $productUrlProvider;

    /**
     * @var CategoryUrlProviderInterface
     */
    protected $categoryUrlProvider;

    /**
     * @var CmsPageUrlProvider
     */
    protected $cmsPageUrlProvider;

    /**
     * @var PurgingConfigProviderInterface
     */
    protected $purgingConfigProvider;

    /**
     * @var array
     */
    protected $purgeBaseUrls;

    /**
     * @var string
     */
    protected $regenBaseUrl;

    /**
     * @var int
     */
    protected $storeViewId;

    /**
     * @var bool
     */
    public $verifyPeer = true;

    /**
     * @var array
     */
    protected $storeCodes = [];

    /**
     * CacheCleaner constructor.
     * @param VarnishUrlRegeneratorFactory $varnishUrlRegeneratorFactory
     * @param VarnishUrlPurgerFactory $varnishUrlPurgerFactory
     * @param LockInterface $lockHandler
     * @param ScopeConfigInterface $scopeConfig
     * @param ProductUrlProviderInterface $productUrlProvider
     * @param CategoryUrlProviderInterface $categoryUrlProvider
     */
    public function __construct(
        VarnishUrlRegeneratorFactory $varnishUrlRegeneratorFactory,
        VarnishUrlPurgerFactory $varnishUrlPurgerFactory,
        LockInterface $lockHandler,
        ScopeConfigInterface $scopeConfig,
        ProductUrlProviderInterface $productUrlProvider,
        CategoryUrlProviderInterface $categoryUrlProvider,
        CmsPageUrlProvider $cmsPageUrlProvider,
        PurgingConfigProviderInterface $purgingConfigProvider
    ) {
        $this->lockHandler = $lockHandler;
        $this->scopeConfig = $scopeConfig;
        $this->productUrlProvider = $productUrlProvider;
        $this->categoryUrlProvider = $categoryUrlProvider;
        $this->cmsPageUrlProvider = $cmsPageUrlProvider;
        $this->purgingConfigProvider = $purgingConfigProvider;

        /** @var VarnishUrlRegeneratorInterface varnishUrlRegenerator */
        $this->varnishUrlRegenerator = $varnishUrlRegeneratorFactory->create();
        /** @var VarnishUrlPurgerInterface varnishUrlPurger */
        $this->varnishUrlPurger = $varnishUrlPurgerFactory->create();
    }

    /**
     * @param int $storeViewId
     */
    public function setStoreViewId(int $storeViewId)
    {
        $this->storeViewId = $storeViewId;
    }

    /**
     * Purge *
     * Regen homepage, categories, products, cms pages
     * @return void
     */
    public function purgeWildcard(): void
    {
        $this->lock();
        $this->addUrlToPurge('*');
        foreach($this->getStoreUrls('') as $storeUrl) {
            $this->addUrlToRegenerate($storeUrl);
        }
        $this->regenerateCategories();
        $this->processProductsRegenerate();
        $this->regenerateCmsPages();
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlPurger->runPurgeQueue();
        $this->varnishUrlRegenerator->runRegenerationQueue();
        $this->unlock();
    }

    /**
     * Purge * without any regeneration
     * Pass through lock
     * @return void
     */
    public function purgeWildcardWithoutRegen(): void
    {
        $this->addUrlToPurge('*');
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlPurger->runPurgeQueue();
    }

    /**
     * Purge homepage, categories, products, cms pages
     * Regen homepage, categories, products, cms pages
     * @return void
     */
    public function purgeAll(): void
    {
        $this->lock();
        $this->addHomepageToPurge();
        $this->processCategoriesPurgeAndRegenerate();
        $this->processCmsPagesPurgeAndRegenerate();
        $this->processProductsPurgeAndRegenerate();
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlPurger->runPurgeQueue();
        $this->varnishUrlRegenerator->runRegenerationQueue();
        $this->unlock();
    }

    /**
     * Generate homepage, categories, products, cms pages
     * @return void
     */
    public function generateAll(): void
    {
        $this->lock();
        foreach($this->getStoreUrls('') as $storeUrl) {
            $this->addUrlToRegenerate($storeUrl);
        }
        $this->regenerateCategories();
        $this->processProductsRegenerate();
        $this->regenerateCmsPages();
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlRegenerator->runRegenerationQueue();
        $this->unlock();
    }

    /**
     * Purge homepage, categories, cms pages
     * Regen homepage, categories, cms pages
     * @return void
     */
    public function purgeGeneral(): void
    {
        $this->lock();
        $this->addHomepageToPurge();
        $this->processCategoriesPurgeAndRegenerate();
        $this->processCmsPagesPurgeAndRegenerate();
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlPurger->runPurgeQueue();
        $this->varnishUrlRegenerator->runRegenerationQueue();
        $this->unlock();
    }

    private function addHomepageToPurge(): void
    {
        foreach($this->getStoreUrls('') as $storeUrl) {
            $this->addUrlToPurge($storeUrl);
            $this->addUrlToRegenerate($storeUrl);
        }
    }

    /**
     * Purge homepage
     * Regen homepage
     * @return void
     */
    public function purgeHomepage(): void
    {
        $this->lock();
        $this->addHomepageToPurge();
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlPurger->runPurgeQueue();
        $this->varnishUrlRegenerator->runRegenerationQueue();
        $this->unlock();
    }

    /**
     * @param string $url
     * @return void
     */
    public function purgeAndRegenerateUrl(string $url): void
    {
        if (substr($url, 0, 1) === '/') {
            $url = ltrim($url, '/');
        }

        $this->addUrlToPurge($url);
        $this->addUrlToRegenerate($url);
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlPurger->runPurgeQueue();
        $this->varnishUrlRegenerator->runRegenerationQueue();
    }

    /**
     * @param $product
     * @return void
     */
    public function purgeProduct($product): void
    {
        $productUrls = $this->getProductUrls($product->getEntityId());
        foreach ($productUrls as $url) {
            if($this->getIsStoreCodeUrlsEnabled()) {
                if (substr($url['request_path'], 0) != "/") {
                    $url['request_path'] = "/" . $url['request_path'];
                }
                $productUrl = $this->getStoreCodeById($url['store_id']) . $url['request_path'];
            }
            $this->addUrlToPurge($productUrl, true);
        }
        $this->varnishUrlPurger->runPurgeQueue();
        $this->varnishUrlRegenerator->runRegenerationQueue();
    }

    /**
     * @return void
     */
    public function purgeAndRegenerateProducts(): void
    {
        $this->lock();
        $this->processProductsPurgeAndRegenerate();
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlPurger->runPurgeQueue();
        $this->varnishUrlRegenerator->runRegenerationQueue();
        $this->unlock();
    }

    /**
     * @return void
     */
    public function purgeAndRegenerateCategories(): void
    {
        $this->lock();
        $this->processCategoriesPurgeAndRegenerate();
        $this->varnishUrlPurger->setVerifyPeer($this->verifyPeer);
        $this->varnishUrlPurger->runPurgeQueue();
        $this->varnishUrlRegenerator->runRegenerationQueue();
        $this->unlock();
    }

    /**
     * @return bool
     */
    public function isLocked(): bool
    {
        return $this->lockHandler->isLocked();
    }

    /**
     * @return string
     */
    public function getLockMessage(): string
    {
        return $this->lockHandler->getLockDate();
    }

    /**
     * @param $relativeUrl
     * @param bool $autoRegenerate
     * @return void
     */
    private function addUrlToPurge($relativeUrl, $autoRegenerate = false): void
    {
        foreach ($this->getPurgeBaseUrls() as $purgeBaseUrl) {
            $url = $purgeBaseUrl . $relativeUrl;
            $this->varnishUrlPurger->addUrlToPurge($url);
            if ($autoRegenerate) {
                $this->addUrlToRegenerate($relativeUrl);
            }
        }
    }

    /**
     * @param $relativeUrl
     * @return void
     */
    private function addUrlToRegenerate($relativeUrl): void
    {
        $url = $this->getRegenBaseUrl() . $relativeUrl;
        $this->varnishUrlRegenerator->addUrlToRegenerate($url);
    }

    /**
     * @return void
     */
    private function regenerateCategories(): void
    {
        $categories = $this->getCategories();
        foreach ($categories as $category) {
            if (empty($category)) {
                continue;
            }

            $categoryUrl = $category['request_path'];

            if($this->getIsStoreCodeUrlsEnabled()) {
                if (substr($category['request_path'], 0, 1) != '/') {
                    $category['request_path'] = '/' . $category['request_path'];
                }
                $categoryUrl = $this->getStoreCodeById($category['store_id']) . $category['request_path'];
            }

            $this->addUrlToRegenerate($categoryUrl);
        }
    }

    /**
     * @return void
     */
    private function regenerateCmsPages(): void
    {
        $cmsPages = $this->getCmsPages();
        foreach ($cmsPages as $cmsPage) {
            if (empty($cmsPage)) {
                continue;
            }

            $cmsPageUrl = $cmsPage['request_path'];

            if($this->getIsStoreCodeUrlsEnabled()) {
                if (substr($cmsPage['request_path'], 0, 1) != '/') {
                    $cmsPage['request_path'] = '/' . $cmsPage['request_path'];
                }
                $cmsPageUrl = $this->getStoreCodeById($cmsPage['store_id']) . $cmsPage['request_path'];
            }

            $this->addUrlToRegenerate($cmsPageUrl);
        }
    }

    /**
     * @return void
     */
    private function processCategoriesPurgeAndRegenerate(): void
    {
        $categories = $this->getCategories();
        foreach ($categories as $category) {
            if (empty($category)) {
                continue;
            }

            $categoryUrl = $category['request_path'];

            if($this->getIsStoreCodeUrlsEnabled()) {
                if (substr($category['request_path'], 0, 1) != "/") {
                    $category['request_path'] = "/" . $category['request_path'];
                }
                $categoryUrl = $this->getStoreCodeById($category['store_id']) . $category['request_path'];
            }

            $this->addUrlToPurge($categoryUrl, true);
        }
    }

    /**
     * @return void
     */
    private function processCmsPagesPurgeAndRegenerate(): void
    {
        $cmsPages = $this->getCmsPages();
        foreach ($cmsPages as $cmsPage) {
            if (empty($cmsPage)) {
                continue;
            }

            $cmsPageUrl = $cmsPage['request_path'];

            if($this->getIsStoreCodeUrlsEnabled()) {
                if (substr($cmsPage['request_path'], 0, 1) != "/") {
                    $cmsPage['request_path'] = "/" . $cmsPage['request_path'];
                }
                $cmsPageUrl = $this->getStoreCodeById($cmsPage['store_id']) . $cmsPage['request_path'];
            }

            $this->addUrlToPurge($cmsPageUrl, true);
        }
    }

    /**
     * @return void
     */
    private function processProductsRegenerate(): void
    {
        $productUrls = $this->getAllProductsUrls();
        foreach ($productUrls as $key => $url) {
            if (empty($url)) {
                continue;
            }

            $productUrl = $url['request_path'];

            if($this->getIsStoreCodeUrlsEnabled()) {
                if (substr($url['request_path'], 0, 1) != "/") {
                    $url['request_path'] = "/" . $url['request_path'];
                }
                $productUrl = $this->getStoreCodeById($url['store_id']) . $url['request_path'];
            }

            $this->addUrlToRegenerate($productUrl);
        }
    }

    /**
     * @return void
     */
    private function processProductsPurgeAndRegenerate(): void
    {
        $productUrls = $this->getAllProductsUrls();
        foreach ($productUrls as $key => $url) {
            if (empty($url)) {
                continue;
            }

            $productUrl = $url['request_path'];

            if($this->getIsStoreCodeUrlsEnabled()) {
                if (substr($url['request_path'], 0, 1) != '/') {
                    $url['request_path'] = "/" . $url['request_path'];
                }
                $productUrl = $this->getStoreCodeById($url['store_id']) . $url['request_path'];
            }
            $this->addUrlToPurge($productUrl, true);
        }
    }

    /**
     * @return array
     */
    private function getAllProductsUrls(): array
    {
        return $this->productUrlProvider->getActiveProductsUrls();
    }

    /**
     * @param $productId
     * @return array
     */
    private function getProductUrls($productId): array
    {
        return $this->productUrlProvider->getProductUrls($productId);
    }

    /**
     * @return array
     */
    private function getCategories(): array
    {
        return $this->categoryUrlProvider->getActiveCategoriesUrls();
    }

    /**
     * @return array
     */
    private function getCmsPages(): array
    {
        return $this->cmsPageUrlProvider->getCmsPageUrls();
    }

    /**
     * @return void
     */
    private function lock(): void
    {
        $this->lockHandler->lock();
    }

    /**
     * @return void
     */
    private function unlock(): void
    {
        $this->lockHandler->unlock();
    }

    /**
     * @return void
     */
    private function setPurgeBaseUrls(): void
    {
        if ($this->purgingConfigProvider->isPurgeCustomHostEnabled()) {
            $this->purgeBaseUrls = $this->purgingConfigProvider->getCustomPurgeHosts();
        } else {
            $baseUrl = $this->scopeConfig->getValue(
                Store::XML_PATH_UNSECURE_BASE_URL,
                ScopeInterface::SCOPE_STORE,
                $this->storeViewId
            );
            $this->purgeBaseUrls = [$baseUrl];
        }

        foreach ($this->purgeBaseUrls as &$purgeBaseUrl) {
            if (substr($purgeBaseUrl, -1) != "/") {
                $purgeBaseUrl .= "/";
            }
        }
    }

    /**
     * @return void
     */
    private function setRegenBaseUrl(): void
    {
        $this->regenBaseUrl = $this->scopeConfig->getValue(
            Store::XML_PATH_UNSECURE_BASE_URL,
            ScopeInterface::SCOPE_STORE,
            $this->storeViewId
        );

        if (substr($this->regenBaseUrl, -1) != "/") {
            $this->regenBaseUrl .= "/";
        }
    }

    /**
     * @return array
     */
    private function getPurgeBaseUrls()
    {
        if (!$this->purgeBaseUrls) {
            $this->setPurgeBaseUrls();
        }
        return $this->purgeBaseUrls;
    }

    /**
     * @return string
     */
    private function getRegenBaseUrl()
    {
        if (!$this->regenBaseUrl) {
            $this->setRegenBaseUrl();
        }
        return $this->regenBaseUrl;
    }

    private function getStoreUrls($url): array
    {
        $urls = [$url];
        if($this->getIsStoreCodeUrlsEnabled()) {
            $storeCodes = $this->getAllStoresCode();
            if($storeCodes) {
                foreach ($storeCodes as $storeCode) {
                    $urls[] = $url . $storeCode . '/';
                }
            }
        }

        return $urls;
    }

    private function getIsStoreCodeUrlsEnabled() {
        return $this->scopeConfig->getValue('web/url/use_store');
    }

    private function getStoreCodeById($storeId) {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $instantiatedClass = $objectManager->create('\Magento\Store\Model\StoreManagerInterface');
        $store = $instantiatedClass->getStore($storeId);
        if($store) {
            return $store->getCode();
        }
        return false;
    }

    private function getAllStoresCode(): array
    {
        if(!isset($this->storeCodes)) {
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            $instantiatedClass = $objectManager->create('\Magento\Store\Model\StoreManagerInterface');
            foreach($instantiatedClass->getStores() as $store) {
                if($store->getIsActive()) {
                    $this->storeCodes[] = $store->getCode();
                }
            }
        }
        return $this->storeCodes;
    }
}
