<?php

namespace EasyNolo\BancaSellaPro\Model;

use Magento\Framework\Convert\DataObject;
use Magento\Quote\Api\Data\PaymentInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Payment;

class Gestpay extends \Magento\Payment\Model\Method\AbstractMethod
{

    const CODE = 'easynolo_bancasellapro';
    const PROD_PAYMENT_URL = 'https://ecomm.sella.it';
    const TEST_PAYMENT_URL = 'https://testecomm.sella.it';
    const PROD_WSDL_URL = 'https://ecomms2s.sella.it';
    const TEST_WSDL_URL = 'https://testecomm.sella.it';
    const PAGE_FOR_PAYMENT = '/pagam/pagam.aspx';
    const PAGE_FOR_3D_AUTH = '/pagam/pagam3d.aspx';

    const MINIMUM_AMOUNT = 0.01;

    protected $_code = self::CODE;

    protected $_canAuthorize = true;
    protected $_canCapture = true;
    protected $_canCapturePartial = true;
    protected $_canRefund = true;
    protected $_canRefundInvoicePartial = true;
    protected $_canVoid = true;

    /**
     * Availability option
     *
     * @var bool
     */
    protected $_isInitializeNeeded = true;

    public function getBaseUrlSella()
    {
        $url = $this->getConfigData('url_live');
        if ($this->getConfigData('debug')) {
            $url = $this->getConfigData('url_test');
        }
        return $url;
    }

    public function getBaseWSDLUrlSella()
    {
        $url = $this->getConfigData('url_live_s2s');
        if ($this->getConfigData('debug')) {
            $url = $this->getConfigData('url_test_s2s');
        }
        return $url;
    }

    public function getRedirectPagePaymentUrl()
    {
        $domain = $this->getBaseUrlSella();
        return $domain . self::PAGE_FOR_PAYMENT;
    }

    public function get3dAuthPageUrl()
    {
        $domain = $this->getBaseUrlSella();
        return $domain . self::PAGE_FOR_3D_AUTH;
    }

    public function isLowRiskProfiledEnabled($order)
    {
        $enableLrp = $this->getConfigData('use_lowriskprofile');
        $lrpThreshold = $this->getLowRiskProfileThreshold();

        if (!empty($enableLrp) && $lrpThreshold > 0.001) {
            $total = (float)$this->getTotalByOrder($order);
            if ($total < $lrpThreshold) {
                return true;
            }
        }

        return false;
    }

    public function getLowRiskProfileThreshold()
    {
        return (float)$this->getConfigData('lrp_threshold');
    }


    public function getLowRiskProfileShopLogin()
    {
        return $this->getConfigData('lrp_merchant_id');
    }

    public function isUseTransactionKeyEnabled()
    {
        $enableTokenization = $this->getConfigData('use_transactionkey');
        if ($enableTokenization) {
            return true;
        }
        return false;
    }

    public function getTransactionKeySiteID()
    {
        return $this->getConfigData('tk_site_id');
    }

    public function getTransactionKeyApiKey()
    {
        return $this->getConfigData('tk_api_key');
    }

    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        if ($quote && $quote->getBaseGrandTotal() < self::MINIMUM_AMOUNT)
            return false;

        if (!$this->getConfigData('merchant_id'))
            return false;

        if (!extension_loaded('soap')) {
            $_helper = $this->_objectManager->create('EasyNolo\BancaSellaPro\Helper\Data');
            $_helper->log('Non è stato possibile creare il client per il webserver - PHP SOAP extension is required.');
            return false;
        }

        return parent::isAvailable($quote);
    }

    public function getTotalByOrder($order)
    {
        $defaultCurrency = $this->getConfigData('currency', \Magento\Store\Model\Store::DEFAULT_STORE_ID);
        $storeCurrency = $this->getConfigData('currency');

        if ($defaultCurrency != $storeCurrency) {
            return $order->getGrandTotal();
        } else {
            return $order->getBaseGrandTotal();
        }

    }

    public function setStatusOrderByS2SRequest($order, $result)
    {
        $payment = $order->getPayment();
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $_helper = $objectManager->create('EasyNolo\BancaSellaPro\Helper\Data');
        $_helper->log('Controllo l\'ordine in base alle risposte della S2S');

        $payment->setAdditionalData(serialize($result->getData()))
        ->setTransactionAdditionalInfo(\Magento\Sales\Model\Order\Payment\Transaction::RAW_DETAILS, $result->getData());

        if ($this->getConfigData('order_status_fraud_gestpay')) {

            $_helper->log('Controllo frode');

            $message = false;
            $total = $this->getTotalByOrder($order);
            $_helper->log('controllo il totale dell\'ordine : ' . $result->getAmount() . ' = ' . round($total, 2));
            if (round($result->getAmount(), 2) != round($total, 2)) {
                // il totatle dell'ordine non corrisponde al totale della transazione
                $message = __('Transaction amount differs from order grand total.');
            }

            if ($result->getAlertCode() != '' && $result->getAlertCode() != 0) {
                $_helper->log('controllo alert della transazione : ' . $result->getAlertCode());
                $message = $result->getAlertDescription();
            }

            if ($message) {
                $_helper->log('rilevata possibile frode: ' . $message);

                $payment->setTransactionId($result->getShopTransactionId())
                    ->setCurrencyCode($order->getBaseCurrencyCode())
                    ->setIsTransactionClosed(0)
                    ->setPreparedMessage($message);
                $order->setState(Order::STATE_PAYMENT_REVIEW);
                $order->setStatus(Order::STATUS_FRAUD);
                $order->addStatusHistoryComment($message, Order::STATUS_FRAUD);

                $order->save();

                return false;
            }

        }

        switch ($result->getTransactionResult()) {

            case 'XX':
                $message = __("Authorizing amount of %1 is pending approval on gateway.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                $order->setState(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
                $status = $order->getConfig()->getStateDefaultStatus(\Magento\Sales\Model\Order::STATE_PENDING_PAYMENT);
                $order->setStatus($status);
                $order->addStatusHistoryComment($message, $status);
                $_helper->log('Pagamento effettuato con bonifico bancario, verificare a mano la transazione');
                $order->addStatusHistoryComment(__('Payment was using bank transfer. Please verify the order status on GestPay.'));
                break;

            case 'OK':
                $riskified_skipped_payments = array('unionpay', 'alipay');
                if ($this->isRedEnabled()):
                    switch ($result->getRedResponseCode()) {
                        case 'ACCEPT':
                            $this->_setOrderPaid($order, $result);
                            break;
                        default:
                            $_helper->log('Pagamento effettuato correttamente ma il check RED è risultato \'' . $result->getRedResponseCode() . '\'. Cambio stato all\'ordine e salvo l\'id della transazione');

                            $message = __("Authorization approved on gateway but RED return with '%1' status. GestPay Transaction ID: %2", $result->getRedResponseCode(), $result->getBankTransactionID());
                            if ($paymentMethod = $result->getPaymentMethod()) {
                                $message .= " (" . $paymentMethod . ")";
                            }

                            $payment->setTransactionId($result->getShopTransactionId())
                                ->setCurrencyCode($order->getBaseCurrencyCode())
                                ->setIsTransactionClosed(0);

                            $status = Order::STATE_PAYMENT_REVIEW;

                            if ($result->getRedResponseCode() == 'DENY')
                                $status = $this->getRedConfigData('deny_order_status');
                            elseif ($result->getRedResponseCode() == 'CHALLENGE')
                                $status = $this->getRedConfigData('challenge_order_status');
                            $order->setState(Order::STATE_PAYMENT_REVIEW);
                            $order->setStatus($status);
                            $order->addStatusHistoryComment($message, $status);

                            $order->save();
                    }
                elseif ($this->isRiskifiedEnabled() && !empty($result->getRiskResponseCode()) && !in_array(strtolower($result->getPaymentMethod()), $riskified_skipped_payments)):
                    switch ($result->getRiskResponseCode()) {
                        case 'approved':
                            $this->_setOrderPaid($order, $result);
                            break;
                        default:
                            $_helper->log('Pagamento effettuato correttamente ma il check Riskified è risultato \'' . $result->getRiskResponseCode() . '\'. Cambio stato all\'ordine e salvo l\'id della transazione');
                            $message = __("Authorization approved on gateway but Riskified return with '%1' status. Response description: %2. GestPay Transaction ID: %3", $result->getRiskResponseCode(), $result->getRiskResponseDescription(), $result->getBankTransactionID());
                            if ($paymentMethod = $result->getPaymentMethod()) {
                                $message .= " (" . $paymentMethod . ")";
                            }
                            
                            $payment->setTransactionId($result->getShopTransactionId())
                                ->setCurrencyCode($order->getBaseCurrencyCode())
                                ->setIsTransactionClosed(0);

                            $status = Order::STATE_PAYMENT_REVIEW;

                            if ($result->getRiskResponseCode() == 'declined')
                                $status = $this->getRiskifiedConfigData('declined_order_status');
                            elseif ($result->getRiskResponseCode() == 'submitted')
                                $status = $this->getRiskifiedConfigData('submitted_order_status');
                            
                            if($status == Order::STATE_CANCELED) {
                                $order->cancel(); // To restore the canceled product quantity
                            }
                            $order->setState(Order::STATE_PAYMENT_REVIEW);
                            $order->setStatus($status);
                            $order->addStatusHistoryComment($message, $status);

                            $order->save();
                            if ($result->getRiskResponseCode() == 'declined') {
                                $this->void($payment); //To perform callDeleteS2S, if riskified declined
                            }
                    }
                else:
                    $this->_setOrderPaid($order, $result);
                endif;
                break;

            case 'KO':
                $_helper->log('Pagamento non andato a buon fine. Cambio stato all\'ordine e salvo l\'id della transazione');
                $message = __("Authorizing amount of %1 is pending approval on gateway.", $order->getBaseCurrency()->formatTxt($order->getBaseGrandTotal()));
                $order->cancel();
                $order->setState($this->getConfigData('order_status_ko_gestpay'));
                $status = $order->getConfig()->getStateDefaultStatus($this->getConfigData('order_status_ko_gestpay'));
                $order->setStatus($status);
                $order->addStatusHistoryComment($message, $status);

                $message = __("Payment attempt has been declined. GestPay Transaction ID: %1", $result->getBankTransactionID());
                if ($paymentMethod = $result->getPaymentMethod()) {
                    $message .= " (" . $paymentMethod . ")";
                }
                $order->addStatusHistoryComment($message);
                break;
        }

        $order->save();
        $_helper->log('Dopo l\'elaborazione della s2s l\'ordine con id: ' . $order->getId() . ' ha state: ' . $order->getState() . ' e status: ' . $order->getStatus());
    }

    public function capture(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $s2s = $objectManager->create('EasyNolo\BancaSellaPro\Model\WS\WS2S');
        $_helper = $objectManager->create('EasyNolo\BancaSellaPro\Helper\Data');
        if (!$this->getConfigData('use_s2s_api')) {
            $message = __('Capture online not allowed. Check payment module configuration "Use S2S Sales API for Capture, Void, Refund actions".');
            throw new \Magento\Framework\Exception\LocalizedException($message);
        }
        $order = $payment->getOrder();
        $params = $_helper->getGestpayParams($order);

        $param = array();
        $param['shopLogin'] =  $params['shopLogin'];
        $param['shopTransID'] =  $params['shopTransactionId'];
        $param['uicCode'] =  $params['uicCode'];
        if(isset($params['apikey']) && !empty(trim($params['apikey']))) {
            $param['apikey'] = $params['apikey'];
        }

        $paymentData = unserialize($payment->getAdditionalData());
        $bankTransId = $paymentData['bank_transaction_i_d'];
        if (!empty($bankTransId) && $bankTransId != $param['shopTransID'])
        {
            $param['bankTransID'] = $bankTransId;
        }
        $param['amount'] = $amount;

        $s2s->capturePayment($payment, $order, $param);      

        return $this;
    }

    public function refund(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $s2s = $objectManager->create('EasyNolo\BancaSellaPro\Model\WS\WS2S');
        $_helper = $objectManager->create('EasyNolo\BancaSellaPro\Helper\Data');
        if (!$this->getConfigData('use_s2s_api')) {
            $message = __('Refund online not allowed. Check payment module configuration "Use S2S Sales API for Capture, Void, Refund actions".');
            throw new \Magento\Framework\Exception\LocalizedException($message);
        }
        $order = $payment->getOrder();
        $params = $_helper->getGestpayParams($order);

        $param = array();
        $param['shopLogin'] =  $params['shopLogin'];
        $param['shopTransactionId'] =  $params['shopTransactionId'];
        $param['uicCode'] =  $params['uicCode'];
        if(isset($params['apikey']) && !empty(trim($params['apikey']))) {
            $param['apikey'] = $params['apikey'];
        }

        $paymentData = unserialize($payment->getAdditionalData());
        $bankTransId = $paymentData['bank_transaction_i_d'];
        if (!empty($bankTransId) && $bankTransId != $param['shopTransactionId'])
        {
            $param['bankTransID'] = $bankTransId;
        }
        $param['amount'] = $amount;

        $s2s->refundPayment($payment, $order, $param);

        return $this;
    }

    public function void(\Magento\Payment\Model\InfoInterface $payment)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $s2s = $objectManager->create('EasyNolo\BancaSellaPro\Model\WS\WS2S');
        $_helper = $objectManager->create('EasyNolo\BancaSellaPro\Helper\Data');
        if (!$this->getConfigData('use_s2s_api')) {
            $message = __('Void online not allowed. Check payment module configuration "Use S2S Sales API for Capture, Void, Refund actions".');
            throw new \Magento\Framework\Exception\LocalizedException($message);
        }
        $order = $payment->getOrder();
        $params = $_helper->getGestpayParams($order);

        $param = array();
        $param['shopLogin'] =  $params['shopLogin'];
        $param['shopTransactionId'] =  $params['shopTransactionId'];
        if(isset($params['apikey']) && !empty(trim($params['apikey']))) {
            $param['apikey'] = $params['apikey'];
        }

        $paymentData = unserialize($payment->getAdditionalData());
        $bankTransId = $paymentData['bank_transaction_i_d'];
        if (!empty($bankTransId) && $bankTransId != $param['shopTransactionId'])
        {
            $param['bankTransactionId'] = $bankTransId;
        }
        $param['CancelReason'] = (string)__('Cancel pre-auth');

        $s2s->voidPayment($payment, $order, $param);

        return $this;
    }


    public function authorize(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $payment->setIsTransactionPending(true);
        return $this;
    }

    private function _setOrderPaid($order, $result)
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $_helper = $objectManager->create('EasyNolo\BancaSellaPro\Helper\Data');
        $payment = $order->getPayment();

        $_helper->log('Pagamento effettuato correttamente. Cambio stato all\'ordine e salvo l\'id della transazione');
        $message = __("Authorization of %1 approved on gateway. GestPay Transaction ID: %2", $order->formatPriceTxt($result->getAmount()), $result->getBankTransactionID());
        if (!empty($result->getPaymentMethod())) {
            $message .= " (" . $result->getPaymentMethod() . ")";
        }

        // create the authorization transaction
        $transId = (string)$result->getBankTransactionID() ? (string)$result->getBankTransactionID() : (string)$result->getShopTransactionId();
        $payment->setData('bankTransactionId', $result->getBankTransactionID());
        
        $payment->setTransactionId($transId);
        $payment->setLastTransId($transId);
        $payment->setCurrencyCode($order->getBaseCurrencyCode());
        $payment->setIsTransactionClosed(0);
        
        $payment->registerAuthorizationNotification($order->getBaseGrandTotal());

        $order->setState($this->getConfigData('order_status_ok_gestpay'));
        $status = $this->getConfigData('order_status_ok_gestpay');
        $order->setStatus($status);
        $order->addStatusHistoryComment($message, $status);

        // perform the capture
        $setOrderAsPaid = true;

        $totalDue = $order->getTotalDue();

        if ($this->getConfigData('payment_action') == \Magento\Payment\Model\Method\AbstractMethod::ACTION_AUTHORIZE_CAPTURE) {
            try {

                if($totalDue != 0){
                    $payment->capture(null);
                    $order->save();
                }
                
            } catch (\Exception $e) {
                $setOrderAsPaid = false;
                $message = __("Failed capture: %1", $e->getMessage());
                $order->setState(\Magento\Sales\Model\Order::STATE_PAYMENT_REVIEW);
                $order->setStatus(\Magento\Sales\Model\Order::STATE_PAYMENT_REVIEW);
                $order->addStatusHistoryComment($message);
            }

            // capture notification, used for capture offline too
            if ($setOrderAsPaid == true) {
                $payment->registerCaptureNotification($order->getBaseGrandTotal(), true);
                $payment->setIsTransactionClosed(1);
            }
        }

        $payment->save();
        $order->save();

        /**
         * we only want to send to customer about new order when there is no redirect to third party
         */
        if ($setOrderAsPaid == true && $order && $order->getCanSendNewEmailFlag() && !$order->getSendEmail()) {
            try {
                $orderSender = $objectManager->create('Magento\Sales\Model\Order\Email\Sender\OrderSender');
                $orderSender->send($order);
            } catch (\Exception $e) {
                $this->_logger->critical($e);
            }
        }
    }

    public function getRedConfigData($field, $storeId = null)
    {
        $path = 'payment/easynolo_bancasellapro_red/' . $field;
        return $this->_scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function isRedEnabled()
    {
        return $this->getRedConfigData('enable');
    }

    public function getRiskifiedConfigData($field, $storeId = null)
    {
        $path = 'payment/easynolo_bancasellapro_riskified/' . $field;
        return $this->_scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function isRiskifiedEnabled()
    {
        return $this->getRiskifiedConfigData('enable');
    }

    /**
     * Assign data to info model instance
     *
     * @param \Magento\Framework\DataObject|mixed $data
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function assignData(\Magento\Framework\DataObject $data)
    {
        /** @var DataObject $info */
        $info = $this->getInfoInstance();

        $details = array('alternative-payment' => null);
        $info->setAdditionalInformation('alternative-payment', null);

        if ($alt = $data->getAdditionalData('alternative_payment')) {
            $alt = json_decode($alt, true);
            foreach ($alt as $v) {
                $details[$v['name']] = $v['value'];
                $info->setAdditionalInformation($v['name'], $v['value']);
            }
        }

        if ($data->getAdditionalData('ask_tokenization')) {
            $info->setAdditionalInformation('ask_tokenization', 1);
        } else {
            $info->setAdditionalInformation('ask_tokenization', 0);
        }

        if (!empty($details)) {
            $info->addData($details);
        }
        parent::assignData($data);
        return $this;
    }

    /**
     * Get config payment action url
     * Used to universalize payment actions when processing payment place
     *
     * @return string
     * @api
     */
    public function getConfigPaymentAction()
    {
        if ($this->getInfoInstance()) {
            $additionalData = $this->getInfoInstance()->getAdditionalInformation();
            if (!empty($additionalData) && !empty($additionalData['alternative-payment'])) {
                return false;
            }
        }

        return $this->getConfigData('iframe') == 1 ? $this->getConfigData('payment_action') : false;
    }

    /**
     * Check if there is redirect URL
     *
     * @return string
     * @api
     */
    public function getOrderPlaceRedirectUrl()
    {
        return true;
    }
}